<?php
// Author: Md. Salaquzzaman
// Date: 2025-06-01
// Description: Controller for listing customers and bills with public access
// Project: Billing System

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Customer;
use App\Models\Bill;
use App\Models\BillProcess;
use App\Models\BillSubject;
use App\Models\Collection;
use App\Models\CollectionChd;
use App\Models\User;
use DB;


class ListController extends Controller
{

    public function publicDashboardStatistics()
    {
        try {
            $tblCustomer = (new Customer)->getTable();
            $tblBill = (new Bill)->getTable();
            $tblBillProcess = (new BillProcess)->getTable();
            $tblCollection = (new Collection)->getTable();
            $tblUser = (new User)->getTable();
            $tblBillSubject = (new BillSubject)->getTable();

            $counts = DB::selectOne("SELECT
                (SELECT COUNT(*) FROM $tblCustomer) as total_customers,
                (SELECT COUNT(*) FROM $tblBill WHERE delete_flag = 0) as total_bills,
                (SELECT COUNT(*) FROM $tblBillProcess WHERE delete_flag = 0) as total_invoices,
                (SELECT COUNT(*) FROM $tblCollection WHERE delete_flag = 0) as total_collections,
                (SELECT COUNT(*) FROM $tblUser) as total_users,
                (SELECT COUNT(*) FROM $tblBillSubject) as total_bill_subjects
            ");

            return response()->json([
                'status' => true,
                'message' => 'Dashboard Statistics',
                'data' => [
                    'total_customers' => $counts->total_customers,
                    'total_bills' => $counts->total_bills,
                    'total_invoices' => $counts->total_invoices,
                    'total_collections' => $counts->total_collections,
                    'total_users' => $counts->total_users,
                    'total_bill_subjects' => $counts->total_bill_subjects,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching data: ' . $e->getMessage()
            ], 400);
        }
    }


    public function publicCustomers(Request $request)
    {

        $perPage = $request->query('per_page');
        $orderBy = $request->query('order_by', 'id');
        $sort = strtolower($request->query('sort', 'asc')) === 'desc' ? 'desc' : 'asc';

        $customerId = $request->query('id');
        $customerName = $request->query('customer_name');
        $phone = $request->query('phone');
        $binNo = $request->query('bin_no');

        $billingName = $request->query('billing_name');
        $billingPhone = $request->query('billing_phone');
        $billingEmail = $request->query('billing_email');

        $vatRegNo = $request->query('vat_reg_no');
        $vatCode = $request->query('vat_code');
        $autoAcNo = $request->query('autoAcNo');
        $status = $request->query('status');

        try {
            $query = Customer::query();

            // Apply filters
            if ($customerId) {
                $query->where('id', $customerId);
            }
            if ($customerName) {
                $query->where('customer_name', 'like', '%' . $customerName . '%');
            }
            if ($phone) {
                $query->where('phone', $phone);
            }
            if ($binNo) {
                $query->where('bin_no', $binNo);
            }
            if ($billingName) {
                $query->where('billing_name', 'like', '%' . $billingName . '%');
            }
            if ($billingPhone) {
                $query->where('billing_phone', $billingPhone);
            }
            if ($billingEmail) {
                $query->where('billing_email', $billingEmail);
            }
            if ($vatRegNo) {
                $query->where('vat_reg_no', $vatRegNo);
            }
            if ($vatCode) {
                $query->where('vat_code', $vatCode);
            }
            if ($autoAcNo) {
                $query->where('autoAcNo', $autoAcNo);
            }
            if ($status) {
                $query->where('status', $status);
            }

            $query->select([
                'id',
                'customer_name',
                'address',
                'phone',
                'bin_no',
                'billing_name',
                'billing_phone',
                'billing_email',
                'vat_reg_no',
                'vat_code',
                'autoAcNo',
                'status',
                'updated_at as created_at',
            ]);

            $query->orderBy($orderBy, $sort);

            // Pagination or get all
            if ($perPage) {
                $customers = $query->paginate($perPage);
                return response()->json([
                    'status' => true,
                    'message' => 'Customer list (paginated)',
                    'data' => $customers->items(),
                    'meta' => [
                        'current_page' => $customers->currentPage(),
                        'last_page' => $customers->lastPage(),
                        'per_page' => $customers->perPage(),
                        'total' => $customers->total()
                    ]
                ]);
            } else {
                $customers = $query->get();
                return response()->json([
                    'status' => true,
                    'message' => 'Customer list (all)',
                    'data' => $customers
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching customers: ' . $e->getMessage()
            ], 400);
        }
    }


    public function publicBills(Request $request)
    {
        $tblBill = (new Bill)->getTable();
        $tblCustomer = (new Customer)->getTable();

        $perPage = $request->query('per_page');
        $orderBy = $request->query('order_by', 'id');
        $sort = strtolower($request->query('sort', 'asc')) === 'desc' ? 'desc' : 'asc';

        $customerId = $request->query('customer_id');
        $billSubjectId = $request->query('bill_subject_id');
        $currency = $request->query('currency');
        $vat_type = $request->query('vat_type');
        $vat_status = $request->query('vat_status');
        $vat_percent = $request->query('vat_percent');
        $billDivName = $request->query('billDivName');
        $bill_group = $request->query('bill_group');
        $bill_type = $request->query('bill_type');
        $status = $request->query('status');
        $vat_refund = $request->query('vat_refund');
        $created_by = $request->query('created_by');
        $start_date = $request->query('start_date');
        $end_date = $request->query('end_date');

        try {
            $query = Bill::query();
            $query->with('customer:id,customer_name,address,phone,bin_no,billing_name,billing_phone,billing_email,vat_reg_no,vat_code');
            $query->with('billSubject:id,bill_subject,gl_subject,gl_code,hs_code');
            $query->where('delete_flag', 0);

            // Apply filters
            if ($customerId) {
                $query->where('customer_id', $customerId);
            }

            if ($billSubjectId) {
                $query->where('bill_subject_id', $billSubjectId);
            }

            if ($currency) {
                $query->where('currency', $currency);
            }

            if ($vat_type) {
                $query->where('vat_type', $vat_type);
            }

            if ($vat_status) {
                $query->where('vat_status', $vat_status);
            }

            if ($vat_percent) {
                $query->where('vat_percent', $vat_percent);
            }

            if ($billDivName) {
                $query->where('billDivName', $billDivName);
            }

            if ($bill_group) {
                $query->where('bill_group', $bill_group);
            }

            if ($bill_type) {
                $query->where('bill_type', $bill_type);
            }

            if ($status) {
                $query->where('status', $status);
            }

            if ($vat_refund) {
                $query->where('vat_refund', $vat_refund);
            }

            if ($created_by) {
                $query->where('created_by', $created_by);
            }

            if ($start_date) {
                $query->where('start_date', $start_date);
            }

            if ($end_date) {
                $query->where('end_date', $end_date);
            }

            $query->select([
                'id',
                'customer_id',
                'bill_subject_id',
                'particulars',
                'currency',
                'currency_rate',
                'qty',
                'unit_price_vat',
                'unit_total',
                'conv_unit_total',
                'total_amount',
                'convertion_total_amount',
                'vat_type',
                'vat_status',
                'vat_percent',
                'vat_amount',
                'conv_vat_amount',
                'total_amount_vat',
                'convertion_total_amount_vat',
                'ref_no',
                'ref_date',
                'installation_date',
                'activation_date',
                'start_date',
                'end_date',
                'department_id',
                'billDivName',
                'bill_group',
                'bill_type',
                'status',
                'vat_refund',
                'updated_at as created_at'
            ]);


            // Sorting logic
            if ($orderBy === 'customer_name') {
                $query->join($tblCustomer, "$tblBill.customer_id", '=', "$tblCustomer.id")
                    ->orderBy("$tblCustomer.customer_name", $sort)
                    ->select("$tblBill.*");
            } else {
                $query->orderBy($orderBy, $sort);
            }

            // Pagination or get all
            if ($perPage) {
                $bills = $query->paginate($perPage);
                return response()->json([
                    'status' => true,
                    'message' => 'Bill list (paginated)',
                    'data' => $bills->items(),
                    'meta' => [
                        'current_page' => $bills->currentPage(),
                        'last_page' => $bills->lastPage(),
                        'per_page' => $bills->perPage(),
                        'total' => $bills->total()
                    ]
                ]);
            } else {
                $bills = $query->get();
                return response()->json([
                    'status' => true,
                    'message' => 'Bill list (all)',
                    'data' => $bills
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching bills: ' . $e->getMessage()
            ], 400);
        }
    }

    public function publicBillSubjects(Request $request)
    {

        $perPage = $request->query('per_page');
        $orderBy = $request->query('order_by', 'id');
        $sort = strtolower($request->query('sort', 'asc')) === 'desc' ? 'desc' : 'asc';

        $id = $request->query('id');
        $billSubject = $request->query('bill_subject');
        $glSubject = $request->query('gl_subject');
        $glCode = $request->query('gl_code');
        $hsCode = $request->query('hs_code');
        $divCode = $request->query('divCode');
        $divName = $request->query('divName');
        $autoAcNo = $request->query('autoAcNo');

        try {
            $query = BillSubject::query();

            if ($id) {
                $query->where('id', $id);
            }
            if ($billSubject) {
                $query->where('bill_subject', 'like', '%' . $billSubject . '%');
            }
            if ($glSubject) {
                $query->where('gl_subject', $glSubject);
            }
            if ($glCode) {
                $query->where('gl_code', $glCode);
            }
            if ($hsCode) {
                $query->where('hs_code', $hsCode);
            }
            if ($divCode) {
                $query->where('divCode', $divCode);
            }
            if ($divName) {
                $query->where('divName', $divName);
            }
            if ($autoAcNo) {
                $query->where('autoAcNo', $autoAcNo);
            }

            $query->select([
                'id',
                'bill_subject',
                'gl_subject',
                'gl_code',
                'hs_code',
                'divCode',
                'divName',
                'autoAcNo',
                'updated_at as created_at'
            ]);


            $query->orderBy($orderBy, $sort);

            // Pagination or get all
            if ($perPage) {
                $billSubjects = $query->paginate($perPage);
                return response()->json([
                    'status' => true,
                    'message' => 'Bill subject list (paginated)',
                    'data' => $billSubjects->items(),
                    'meta' => [
                        'current_page' => $billSubjects->currentPage(),
                        'last_page' => $billSubjects->lastPage(),
                        'per_page' => $billSubjects->perPage(),
                        'total' => $billSubjects->total()
                    ]
                ]);
            } else {
                $billSubjects = $query->get();
                return response()->json([
                    'status' => true,
                    'message' => 'Bill subject list (all)',
                    'data' => $billSubjects
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching bill subjects: ' . $e->getMessage()
            ], 400);
        }
    }


    public function publicInvoices(Request $request)
    {

        $perPage = $request->query('per_page');
        $orderBy = $request->query('order_by', 'id');
        $sort = strtolower($request->query('sort', 'asc')) === 'desc' ? 'desc' : 'asc';

        $id = $request->query('id');
        $invoiceNo = $request->query('invoice_no');
        $processNoFormat = $request->query('process_no_format');
        $customerId = $request->query('customer_id');
        $billSubjectId = $request->query('bill_subject_id');
        $currency = $request->query('currency');

        $vat_type = $request->query('vat_type');
        $vat_status = $request->query('vat_status');
        $vat_percent = $request->query('vat_percent');
        $vatRefund = $request->query('vat_refund');
        $paymentStatus = $request->query('payment_status');
        $paymentType = $request->query('payment_type');

        $departmentId = $request->query('department_id');
        $billDivName = $request->query('billDivName');
        $bill_group = $request->query('bill_group');
        $bill_type = $request->query('bill_type');

        $month = $request->query('month');
        $year = $request->query('year');
        $invoiceDate = $request->query('invoice_date');
        $start_date = $request->query('start_date');
        $end_date = $request->query('end_date');

        try {

            $query = BillProcess::query();
            $query->with('customer:id,customer_name,address,phone,bin_no,billing_name,billing_phone,billing_email,vat_reg_no,vat_code');
            $query->with('billSubject:id,bill_subject,gl_subject,gl_code,hs_code');
            $query->where('delete_flag', 0);


            if ($id) {
                $query->where('id', $id);
            }
            if ($invoiceNo) {
                $query->where('invoice_no', $invoiceNo);
            }
            if ($processNoFormat) {
                $query->where('process_no_format', $processNoFormat);
            }
            if ($customerId) {
                $query->where('customer_id', $customerId);
            }
            if ($billSubjectId) {
                $query->where('bill_subject_id', $billSubjectId);
            }
            if ($currency) {
                $query->where('currency', $currency);
            }
            if ($vat_type) {
                $query->where('vat_type', $vat_type);
            }
            if ($vat_status) {
                $query->where('vat_status', $vat_status);
            }
            if ($vat_percent) {
                $query->where('vat_percent', $vat_percent);
            }
            if ($vatRefund) {
                $query->where('vat_refund', $vatRefund);
            }
            if ($paymentStatus) {
                $query->where('payment_status', $paymentStatus);
            }
            if ($paymentType) {
                $query->where('payment_type', $paymentType);
            }
            if ($departmentId) {
                $query->where('department_id', $departmentId);
            }
            if ($billDivName) {
                $query->where('billDivName', $billDivName);
            }
            if ($bill_group) {
                $query->where('bill_group', $bill_group);
            }
            if ($bill_type) {
                $query->where('bill_type', $bill_type);
            }
            if ($month) {
                $query->where('month', $month);
            }
            if ($year) {
                $query->where('year', $year);
            }
            if ($invoiceDate) {
                $query->where('invoice_date', $invoiceDate);
            }
            if ($start_date) {
                $query->where('start_date', $start_date);
            }
            if ($end_date) {
                $query->where('end_date', $end_date);
            }

            $query->select([
                'id',
                'bill_id',
                'invoice_no',
                'invoice_date',
                'process_no_format',
                'customer_id',
                'bill_subject_id',
                'bill_group',
                'bill_subject_desc',
                'particulars',
                'currency',
                'currency_rate',
                'qty',
                'unit_price_vat',
                'unit_total',
                'conv_unit_total',
                'total_amount',
                'convertion_total_amount',
                'vat_type',
                'vat_status',
                'vat_percent',
                'vat_amount',
                'conv_vat_amount',
                'total_amount_vat',
                'convertion_total_amount_vat',
                'due',
                'payment_status',
                'ref_no',
                'ref_date',
                'installation_date',
                'activation_date',
                'start_date',
                'end_date',
                'department_id',
                'billDivName',
                'bill_type',
                'vat_refund',
                'month',
                'year',
                'remarks',
                'updated_at as created_at'
            ]);

            $query->orderBy($orderBy, $sort);

            // Pagination or get all
            if ($perPage) {
                $invoices = $query->paginate($perPage);
                return response()->json([
                    'status' => true,
                    'message' => 'Invoice list (paginated)',
                    'data' => $invoices->items(),
                    'meta' => [
                        'current_page' => $invoices->currentPage(),
                        'last_page' => $invoices->lastPage(),
                        'per_page' => $invoices->perPage(),
                        'total' => $invoices->total()
                    ]
                ]);
            } else {
                $invoices = $query->get();
                return response()->json([
                    'status' => true,
                    'message' => 'Invoice list (all)',
                    'data' => $bills
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching invoices: ' . $e->getMessage()
            ], 400);
        }
    }


    public function publicCollections(Request $request)
    {

        $perPage = $request->query('per_page');
        $orderBy = $request->query('order_by', 'id');
        $sort = strtolower($request->query('sort', 'asc')) === 'desc' ? 'desc' : 'asc';

        $id = $request->query('id');
        $customer_id = $request->query('customer_id');
        $ref_no = $request->query('ref_no');
        $coll_date = $request->query('coll_date');
        $instrument_mode = $request->query('instrument_mode');
        $instrument_date = $request->query('instrument_date');
        $bank_id = $request->query('bank_id');
        $money_receipt_no = $request->query('money_receipt_no');
        $money_receipt_date = $request->query('money_receipt_date');
        $bank_slip_no = $request->query('bank_slip_no');
        $bank_deposit_date = $request->query('bank_deposit_date');
        $bank_slip_no = $request->query('bank_slip_no');

        try {

            $query = Collection::query();
            $query->with('customer:id,customer_name,address,phone,bin_no,billing_name,billing_phone,billing_email,vat_reg_no,vat_code');
            $query->with('customerBank:id,bank_id,bank_name,address,phone,email');
            $query->where('delete_flag', 0);

            if ($id) {
                $query->where('id', $id);
            }

            if ($customer_id) {
                $query->where('customer_id', $customer_id);
            }

            if ($ref_no) {
                $query->where('ref_no', $ref_no);
            }

            if ($coll_date) {
                $query->whereDate('coll_date', $coll_date);
            }

            if ($instrument_mode) {
                $query->where('instrument_mode', $instrument_mode);
            }

            if ($instrument_date) {
                $query->whereDate('instrument_date', $instrument_date);
            }

            if ($bank_id) {
                $query->where('bank_id', $bank_id);
            }

            if ($money_receipt_no) {
                $query->where('money_receipt_no', $money_receipt_no);
            }

            if ($money_receipt_date) {
                $query->whereDate('money_receipt_date', $money_receipt_date);
            }

            if ($bank_slip_no) {
                $query->where('bank_slip_no', $bank_slip_no);
            }

            if ($bank_deposit_date) {
                $query->whereDate('bank_deposit_date', $bank_deposit_date);
            }

            $query->select([
                'id',
                'customer_id',
                'ref_no',
                'coll_date',
                'instrument_mode',
                'instrument_no',
                'instrument_date',
                'bank_id',
                'money_receipt_no',
                'money_receipt_date',
                'bank_slip_no',
                'bank_deposit_date',
                'received_amount',
                'tds_amount',
                'vat_amount',
                'adjust_amount',
                'advance_amount',
                'total_coll_amount',
                'account_name as depository_account',
                'created_at',
            ]);


            if($orderBy=='ref_no'){
                $query->orderByRaw('CAST(ref_no AS UNSIGNED) DESC');
            }else{
                $query->orderBy($orderBy, $sort);
            }

            // Pagination or get all
            if ($perPage) {
                $collections = $query->paginate($perPage);
                return response()->json([
                    'status' => true,
                    'message' => 'Collection list (paginated)',
                    'data' => $collections->items(),
                    'meta' => [
                        'current_page' => $collections->currentPage(),
                        'last_page' => $collections->lastPage(),
                        'per_page' => $collections->perPage(),
                        'total' => $collections->total()
                    ]
                ]);
            } else {
                $collections = $query->get();
                return response()->json([
                    'status' => true,
                    'message' => 'Collection list (all)',
                    'data' => $collections
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching collections: ' . $e->getMessage()
            ], 400);
        }
    }


    public function publicUsers(Request $request)
    {

        $perPage = $request->query('per_page');
        $orderBy = $request->query('order_by', 'id');
        $sort = strtolower($request->query('sort', 'asc')) === 'desc' ? 'desc' : 'asc';

        $id = $request->query('id');
        $name = $request->query('name');
        $email = $request->query('email');
        $phone = $request->query('phone');
        $type = $request->query('type');
        $employeeId = $request->query('employee_id');

        $divCode = $request->query('divCode');
        $activeStatus = $request->query('active_status');
        $isActive = $request->query('is_active');

        try {
            $query = User::query();

            // Apply filters
            if ($id) {
                $query->where('id', $id);
            }
            if ($name) {
                $query->where('name', 'like', '%' . $name . '%');
            }
            if ($email) {
                $query->where('email', $email);
            }
            if ($phone) {
                $query->where('phone', $phone);
            }
            if ($type) {
                $query->where('type', $type);
            }
            if ($employeeId) {
                $query->where('employee_id', $employeeId);
            }
            if ($divCode) {
                $query->where('divCode', $divCode);
            }
            if ($activeStatus) {
                $query->where('active_status', $activeStatus);
            }
            if ($isActive) {
                $query->where('is_active', $isActive);
            }

            $query->select([
                'id',
                'name',
                'email',
                'phone',
                'type',
                'avatar',
                'address',
                'active_status',
                'is_active',
                'updated_at as created_at'
            ]);

            $query->orderBy($orderBy, $sort);

            // Pagination or get all
            if ($perPage) {
                $users = $query->paginate($perPage);
                return response()->json([
                    'status' => true,
                    'message' => 'User list (paginated)',
                    'data' => $users->items(),
                    'meta' => [
                        'current_page' => $users->currentPage(),
                        'last_page' => $users->lastPage(),
                        'per_page' => $users->perPage(),
                        'total' => $users->total()
                    ]
                ]);
            } else {
                $users = $query->get();
                return response()->json([
                    'status' => true,
                    'message' => 'User list (all)',
                    'data' => $users
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching users: ' . $e->getMessage()
            ], 400);
        }
    }


    public function publicBillGroup()
    {

        $tblBill = (new Bill)->getTable();
        $tblBillProcess = (new BillProcess)->getTable();

        try {

            // $billGroups = DB::select("SELECT DISTINCT bill_group
            //     FROM (
            //         SELECT b.bill_group FROM $tblBill b
            //         UNION
            //         SELECT b.bill_group FROM $tblBillProcess b
            //     ) AS combined
            //     WHERE bill_group IS NOT NULL AND TRIM(bill_group) <> ''
            //     ORDER BY bill_group");

            $billGroups = DB::select("SELECT DISTINCT bill_group
                FROM (
                    SELECT bill_group FROM $tblBill WHERE bill_group IS NOT NULL AND bill_group != ''
                    UNION ALL
                    SELECT bill_group FROM $tblBillProcess WHERE bill_group IS NOT NULL AND bill_group != ''
                ) AS combined
                ORDER BY bill_group
            ");

            return response()->json([
                'status' => true,
                'message' => 'Bill group (all)',
                'data' => $billGroups
            ]);


        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching bill group: ' . $e->getMessage()
            ], 400);
        }
    }

    public function publicBillDepartment()
    {

        $tblBill = (new Bill)->getTable();

        try {

            $billDepartmnts = DB::select("SELECT b.department_id,b.billDivName
                FROM $tblBill b
                GROUP BY b.department_id,b.billDivName ORDER BY b.billDivName");

                return response()->json([
                    'status' => true,
                    'message' => 'Bill department (all)',
                    'data' => $billDepartmnts
                ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'Error fetching bill department: ' . $e->getMessage()
            ], 400);
        }
    }



}
