<?php

namespace App\Http\Controllers;

use Auth;
use App\Models\Bill;
use App\Models\User;
use App\Models\GlHead;
use App\Models\Customer;
use App\Models\Settings;
use App\Models\VatSetup;
use App\Models\Department;
use App\Models\AttentionPerson;
use App\Models\BillProcess;
use App\Models\BillRevision;
use App\Models\BillSubject;
use App\Models\BillNote;
use App\Models\UserDept;
use App\Models\CustomerLedger;
use App\Helpers\LogActivityHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class BillController extends Controller
{


    public function index(Request $request)
    {
        if (!auth()->user()->can('View Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 50);

        $query = Bill::query();
        $query->select([
            'id',
            'customer_id',
            'particulars',
            'bill_subject_id',
            'bill_group',
            'qty',
            'total_amount',
            'vat_amount',
            'total_amount_vat',
            'bill_type',
            'status',
            'bill_note',
            'approval'
        ]);

        $query->where('approval', '1');
        $query->where('delete_flag', 0);

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('bill_no')) {
            $query->where('id', $request->bill_no);
        }
        $query->with(['billSubject:id,bill_subject']);

        $bills = $query->orderBy('id', 'desc')->paginate($perPage)->appends($request->except('page'));

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('bill.index', compact('bills', 'perPage', 'customers'));
    }

    public function proposalInfo(Request $request)
    {
        if (!auth()->user()->can('Proposal Info')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 50);


        $user = auth()->user();
        if (in_array("company", $user->getRoleNames()->toArray())) {
            $bills = Bill::where('delete_flag', 0)
                ->orderBy('id', 'desc')->paginate($perPage);
        } else {
            $bills = Bill::where('delete_flag', 0)
                ->where('created_by', auth()->id())
                ->orderBy('id', 'desc')->paginate($perPage);
        }

        return view('bill.proposal-info', compact('bills', 'perPage'));
    }



    public function create()
    {
        if (!auth()->user()->can('Create Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $departments = [];
        $billSubjects = BillSubject::select('id', 'bill_subject')->orderBy('bill_subject')->get();
        $vatsetup = VatSetup::select('id', 'vat')->orderBy(DB::raw('CAST(vat AS SIGNED)'), 'asc')->get();
        $customers = Customer::where('status','Active')->orderBy('customer_name', 'asc')->get();

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        if($token==''){
            redirect()->route('dashboard')->with('error', 'API Connection Timeout');
        }

        //get department list from api - salaquzzaman @ 05-06-2024
        try{
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        }catch (\Exception $e) {
            // Log::error($e->getMessage());
            if($e->getMessage()){
                redirect()->route('dashboard')->with('error', 'API Connection Timeout');
            }
        }

        return view('bill.create', compact('departments','billSubjects','customers', 'vatsetup'));
    }

    public function store(Request $request)
    {

        if (!auth()->user()->can('Create Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'qty' => 'required',
            'unit_price_vat' => 'required',
            'total_amount' => 'required',
            'currency' => 'required',
            'currency_rate' => 'required|numeric|min:1',
            // 'work_order' => 'required|file|mimes:jpg,jpeg,png,pdf',
            // 'installation_uat' => 'required|file|mimes:jpg,jpeg,png,pdf',
            // 'delivery_challan' => 'file|mimes:jpg,jpeg,png,pdf',
        ]);

        if ($validator->fails()) {
            return redirect()->route('bills.create')
                ->with('error', 'Missing required fields.');
        }

        // Check customer available or not
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        }

        // Proceed to bill create
        $dateFields = ['ref_date', 'installation_date', 'activation_date', 'start_date', 'end_date', 'increase_start_date'];
        $data = $request->all();

        $data['customer_id'] = $custPKID;

        $data['conv_unit_total'] = round(($request->unit_total * $request->currency_rate),2);
        $data['convertion_total_amount'] = round(($request->total_amount * $request->currency_rate),2);
        $data['convertion_total_amount_vat'] = round(($request->total_amount_vat * $request->currency_rate),2);
        $data['conv_vat_amount'] = round(($request->vat_amount * $request->currency_rate),2);

        foreach ($dateFields as $field) {
            if ($request->filled($field)) {
                $data[$field] = $this->normalizeDate($request->$field);
            }
        }

        // Attention Person
        $attention_id = $request->attention_id;
        if($attention_id!='' && !empty($attenInfo=AttentionPerson::find($attention_id))){
            $data['attention_id'] = $attenInfo->id;
            $data['attention_person'] = $attenInfo->attention_name;
        }
        $data['bin_number'] = null;

        if ($request->hasFile('work_order')) {
            $file = $request->file('work_order');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs ('public/workorders', $filename);
            $data['workorder_file'] = $filename;
        }

        if ($request->hasFile('installation_uat')) {
            $file = $request->file('installation_uat');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs ('public/installation', $filename);
            $data['instalation_file'] = $filename;
        }

        if ($request->hasFile('delivery_challan')) {
            $file = $request->file('delivery_challan');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs ('public/deliverychallan', $filename);
            $data['deliverychallan_file'] = $filename;
        }

        $data['delete_flag'] = '0';
        $data['approval'] = 2;

        $billInfo = Bill::create($data + ['created_by' => auth()->id()]);
        if($billInfo){
            $billInfo->update(['bill_no_x'=>$billInfo->id]);
            LogActivityHelper::addToLog('Bill Create','Bill',$billInfo->id);
            return redirect()->back()->with('success', 'Bill created successfully.');
        }else{
            return redirect()->back()->with('error', 'Bill not created.');
        }

    }

    public function show(Bill $bill)
    {
        return view('bill.show', compact('bill'));
    }

    public function getOrderSlNo(Request $request)
    {
        $customerId = $request->input('customer_id');
        $billSubjectId = $request->input('bill_subject_id');
        $vatRefund = $request->input('vat_refund');
        $initialCustomerId = $request->input('initial_customer_id');
        $initialBillSubjectId = $request->input('initial_bill_subject_id');
        $initialVatRefund = $request->input('initial_vat_refund');

        $customerIds = is_array($customerId) ? $customerId : [$customerId];

        $findcustomer = DB::table('billing_customers')
            ->whereIn('autoAcNo', $customerIds)
            ->pluck('id');

        $nextOrderSlNo = 1;

        if ($findcustomer->isNotEmpty()) {
            $rowCount = Bill::whereIn('customer_id', $findcustomer)
                ->where('bill_subject_id', $billSubjectId)
                ->where('vat_refund', $vatRefund)
                ->count();

            if ($customerId == $initialCustomerId && $billSubjectId == $initialBillSubjectId && $vatRefund == $initialVatRefund) {
                $nextOrderSlNo = $request->input('current_order_sl_no', 1);
            } else {
                $nextOrderSlNo = $rowCount + 1;
            }
        }

        return response()->json(['next_order_sl_no' => $nextOrderSlNo]);
    }


    // This page for viewing purpose
    public function awaitingEdit(Bill $bill, $id)
    {
        if (!auth()->user()->can('Edit Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bill = Bill::find($id);
        $showAwaitingBtn = true;
        if($bill->approval==''){
            $showAwaitingBtn = false;
        }

        return view('bill.awaiting-edit-show', compact('bill','showAwaitingBtn'));

    }

    // This page for viewing purpose
    public function proposalInfoShow(Bill $bill, $id)
    {
        $bill = Bill::find($id);
        $showAwaitingBtn = false;

        return view('bill.awaiting-edit-show', compact('bill','showAwaitingBtn'));

    }


    public function edit(Bill $bill)
    {
        if (!auth()->user()->can('Edit Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $departments = [];
        // $billSubjects = BillSubject::select('id', 'bill_subject','divName','hs_code')->where('divCode', $bill->department_id)->get();
        $billSubjects = BillSubject::select('id', 'bill_subject','divName','hs_code')->orderBy('bill_subject')->get();
        $vatsetup = VatSetup::select('id', 'vat')->orderBy(DB::raw('CAST(vat AS SIGNED)'), 'asc')->get();
        $customers = Customer::where('status','Active')->orderBy('customer_name', 'asc')->get();

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        // Get department list from API
        try {
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        } catch (\Exception $e) {
            // Log::error($e->getMessage());
        }

        $autoAcNo = '';
        $binNo = '';
        if ($bill->customer_id && ($custInfo = Customer::find($bill->customer_id))) {
            $autoAcNo = $custInfo->autoAcNo;
            $binNo = $custInfo->bin_no;
        }

        return view('bill.edit', compact('bill', 'departments', 'billSubjects', 'customers', 'autoAcNo', 'vatsetup','binNo'))
            ->with('current_order_sl_no', $bill->order_sl_no);
    }


    public function proposalInfoEdit($id)
    {
        if (!auth()->user()->can('Proposal Info')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bill = Bill::find($id);

        $departments = [];
        // $billSubjects = BillSubject::select('id', 'bill_subject','divName','hs_code')->where('divCode', $bill->department_id)->get();
        $billSubjects = BillSubject::select('id', 'bill_subject','divName','hs_code')->orderBy('bill_subject')->get();
        $vatsetup = VatSetup::select('id', 'vat')->orderBy(DB::raw('CAST(vat AS SIGNED)'), 'asc')->get();
        $customers = Customer::where('status','Active')->orderBy('customer_name', 'asc')->get();

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        // Get department list from API
        try {
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        } catch (\Exception $e) {
            // Log::error($e->getMessage());
        }

        $autoAcNo = '';
        $binNo = '';
        if ($bill->customer_id && ($custInfo = Customer::find($bill->customer_id))) {
            $autoAcNo = $custInfo->autoAcNo;
            $binNo = $custInfo->bin_no;
        }

        return view('bill.proposal-info-edit', compact('bill', 'departments', 'billSubjects', 'customers', 'autoAcNo', 'vatsetup','binNo'))
            ->with('current_order_sl_no', $bill->order_sl_no);
    }

    public function update(Request $request, Bill $bill)
    {

        if (!auth()->user()->can('Edit Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'qty' => 'required',
            'unit_price_vat' => 'required',
            'total_amount' => 'required',
            'bill_subject_id' => 'required',
        ]);

        if ($validator->fails()) {
            return redirect()->route('bills.index')
                ->with('error', 'Missing required fields.');
        }

        // Check if customer exists
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        }

        // Prepare data for bill update
        $dateFields = ['ref_date', 'installation_date', 'activation_date', 'start_date', 'end_date', 'increase_start_date'];
        $data = $request->all();
        $data['customer_id'] = $custPKID;
        $data['bill_editor_id'] = auth()->id();
        $data['bill_edited'] = 1;

        // Update department_id in customer
        Customer::find($custPKID)->update(['divCode' => $request->department_id]);

        $billSubjectId = $request->bill_subject_id;
        if($billSubjectId && $subInfo = BillSubject::find($billSubjectId)->first()){
            $data['hs_code'] = $subInfo->hs_code;
        }

        $data['conv_unit_total'] = round(($request->unit_total * $request->currency_rate),2);
        $data['convertion_total_amount'] = round(($request->total_amount * $request->currency_rate),2);
        $data['convertion_total_amount_vat'] = round(($request->total_amount_vat * $request->currency_rate),2);
        $data['conv_vat_amount'] = round(($request->vat_amount * $request->currency_rate),2);

        foreach ($dateFields as $field) {
            if ($request->filled($field)) {
                $data[$field] = $this->normalizeDate($request->$field);
            }
        }

        $data['bin_number'] = null;

        // Handle work_order file
        if ($request->hasFile('work_order')) {
            if ($bill->workorder_file) {
                Storage::disk('public')->delete('workorders/' . $bill->workorder_file);
            }
            $file = $request->file('work_order');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs('public/workorders', $filename);
            $bill->workorder_file = $filename;
        }

        // Handle installation_uat file
        if ($request->hasFile('installation_uat')) {
            if ($bill->instalation_file) {
                Storage::disk('public')->delete('installation/' . $bill->instalation_file);
            }
            $file = $request->file('installation_uat');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs('public/installation', $filename);
            $bill->instalation_file = $filename;
        }

        // Handle delivery_challan file
        if ($request->hasFile('delivery_challan')) {
            if ($bill->deliverychallan_file) {
                Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
            }
            $file = $request->file('delivery_challan');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs('public/deliverychallan', $filename);
            $bill->deliverychallan_file = $filename;
        } else {
            if ($request->input('delete_delivery_challan') == '1') {
                if ($bill->deliverychallan_file) {
                    Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
                }
                $bill->deliverychallan_file = null;
            }
        }

        if ($request->input('vat_status') == '0') { // No selected
            $data['vat_percent'] = null;
        }

        // Approval Setting
        if($request->approve==1){ // if approve
            $data['approval'] = 1;
            $data['approved_by'] = auth()->id();
        }else if($request->approve==2){ // if request to approval
            $data['approval'] = null;
        }else{}

        $bill->update($data + ['updated_by' => auth()->id()]);
        LogActivityHelper::addToLog('Bill Edit','Bill',$bill->id);

        return redirect()->route('bills.pending')
            ->with('success', 'Bill updated successfully.');

    }

    // proposal info update
    public function proposalInfoUpdate(Request $request, $id)
    {

        if (!auth()->user()->can('Proposal Info')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'qty' => 'required',
            'unit_price_vat' => 'required',
            'total_amount' => 'required',
        ]);

        if ($validator->fails()) {
            return redirect()->route('bills.index')
                ->with('error', 'Missing required fields.');
        }

        $bill = Bill::find($id);

        // Check if customer exists
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        }

        // Prepare data for bill update
        $dateFields = ['ref_date', 'installation_date', 'activation_date', 'start_date', 'end_date', 'increase_start_date'];
        $data = $request->all();
        $data['customer_id'] = $custPKID;
        $data['bill_editor_id'] = auth()->id();
        $data['bill_edited'] = 1;

        // Update department_id in customer
        Customer::find($custPKID)->update(['divCode' => $request->department_id]);

        $billSubjectId = $request->bill_subject_id;
        if($billSubjectId && $subInfo = BillSubject::find($billSubjectId)->first()){
            $data['hs_code'] = $subInfo->hs_code;
        }

        $data['conv_unit_total'] = round(($request->unit_total * $request->currency_rate),2);
        $data['convertion_total_amount'] = round(($request->total_amount * $request->currency_rate),2);
        $data['convertion_total_amount_vat'] = round(($request->total_amount_vat * $request->currency_rate),2);
        $data['conv_vat_amount'] = round(($request->vat_amount * $request->currency_rate),2);

        foreach ($dateFields as $field) {
            if ($request->filled($field)) {
                $data[$field] = $this->normalizeDate($request->$field);
            }
        }

        $data['bin_number'] = null;

        // Handle work_order file
        if ($request->hasFile('work_order')) {
            if ($bill->workorder_file) {
                Storage::disk('public')->delete('workorders/' . $bill->workorder_file);
            }
            $file = $request->file('work_order');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs('public/workorders', $filename);
            $bill->workorder_file = $filename;
        }

        // Handle installation_uat file
        if ($request->hasFile('installation_uat')) {
            if ($bill->instalation_file) {
                Storage::disk('public')->delete('installation/' . $bill->instalation_file);
            }
            $file = $request->file('installation_uat');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs('public/installation', $filename);
            $bill->instalation_file = $filename;
        }

        // Handle delivery_challan file
        if ($request->hasFile('delivery_challan')) {
            if ($bill->deliverychallan_file) {
                Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
            }
            $file = $request->file('delivery_challan');
            $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
            $file->storeAs('public/deliverychallan', $filename);
            $bill->deliverychallan_file = $filename;
        } else {
            if ($request->input('delete_delivery_challan') == '1') {
                if ($bill->deliverychallan_file) {
                    Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
                }
                $bill->deliverychallan_file = null;
            }
        }

        if ($request->input('vat_status') == '0') { // No selected
            $data['vat_percent'] = null;
        }

        // Approval Setting
        $data['approval'] = 2;
        $data['approved_by'] = null;
        $data['awaiting_approved_by'] = null;
        $data['bill_editor_id'] = null;
        $data['bill_edited'] = 0;

        $bill->update($data + ['updated_by' => auth()->id()]);
        LogActivityHelper::addToLog('Bill Edit','Bill',$bill->id);

        return redirect()->route('bills.proposal-info')
            ->with('success', 'Bill updated successfully.');

    }

    public function proposalInfoDelete(Request $request){

        if (!auth()->user()->can('Proposal Info')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bill = Bill::findOrFail($request->deletedID);
        if($bill){
            $process = BillProcess::where('bill_id',$bill->id)->where('delete_flag',0)->first();
            if($process){
                return redirect()->route('bills.proposal-info')
                    ->with('success', 'Bill not deleted, already in processed');
            }else{
                if ($bill->workorder_file) {
                    Storage::disk('public')->delete('workorders/' . $bill->workorder_file);
                }
                if ($bill->instalation_file) {
                    Storage::disk('public')->delete('installation/' . $bill->instalation_file);
                }
                if ($bill->deliverychallan_file) {
                    Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
                }

                $bill->delete();
                LogActivityHelper::addToLog('Bill Delete','Bill',$bill->id);

                return redirect()->route('bills.proposal-info')
                    ->with('success', 'Bill deleted successfully.');
            }
        }

    }


    public function destroy(Bill $bill) // NOT UPDATED, NOT WORKING
    {
        if (!auth()->user()->can('Delete Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bill->delete();

        return redirect()->route('bills.index')
            ->with('success', 'Bill deleted successfully.');
    }


    public function billDelete($id,$type)
    {
        if (!auth()->user()->can('Delete Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $bill = Bill::findOrFail($id);

        if ($bill->workorder_file) {
            Storage::disk('public')->delete('workorders/' . $bill->workorder_file);
        }
        if ($bill->instalation_file) {
            Storage::disk('public')->delete('installation/' . $bill->instalation_file);
        }
        if ($bill->deliverychallan_file) {
            Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
        }

        $bill->delete();
        LogActivityHelper::addToLog('Bill Delete','Bill',$bill->id);

        if($type=='pending'){
            return redirect()->route('bills.pending')->with('success', 'Bill Deleted successfully.');
        }else{
            return redirect()->route('bills.rejected')->with('success', 'Bill Deleted successfully.');
        }

    }

    public function billEditBack(Request $request, $id)
    {
        $bill = Bill::find($id);
        $bill->approval = 0;
        $bill->save();

        $rejectNote = $request->reject_note_text;

        $billNote              = new BillNote();
        $billNote->bill_id     = $bill->id;
        $billNote->note        = $rejectNote;
        $billNote->created_by  = auth()->id();
        $billNote->save();

        try {
            $this->sendRejectEmail($bill, $rejectNote);
        } catch (\Exception $e) {

        }

        return redirect()->route('bills.pending')->with('success', 'Bill Status Changed & Deleted Successfully.');
    }

    public function billEditBackEditor(Request $request, $id)
    {
        $bill = Bill::find($id);
        $bill->bill_editor_id = auth()->id();
        $bill->bill_edited = 0;
        $bill->approval = 2;
        $bill->save();

        $rejectNote = $request->reject_note_text;

        $billNote              = new BillNote();
        $billNote->bill_id     = $bill->id;
        $billNote->note        = $rejectNote;
        $billNote->created_by  = auth()->id();
        $billNote->save();

        // $this->sendRejectEmail($bill, $rejectNote);

        return redirect()->route('bills.pending')->with('success', 'Bill Rejected Successfully.');
    }

    public function showBillNote($id)
    {
        $bill = Bill::find($id);
        if (!auth()->user()->can('View Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        return view('bill.show-bill-note', compact('bill'));
    }


    public function awaitingBills(Request $request)
    {

        // Get the number of entries per page from the request, default to 25
        $perPage = $request->input('per_page', 50);

        // Fetch bills with pagination (Approval-2 = Initial Pending Stage)

        $user = auth()->user();
        if (in_array("company", $user->getRoleNames()->toArray())) {
            $bills = Bill::where(function ($query) {
                $query->where('approval', 2)
                    ->orWhereNull('approval');
                })
                ->where('delete_flag', 0)
                ->orderBy('id', 'desc')
                ->paginate($perPage);
        } else {
            // $bills = Bill::where('approval', 2)
            //     ->where('approval', null)
            //     ->where('delete_flag', 0)
            //     ->where('department_head', auth()->id())
            //     ->orderBy('id', 'desc')
            //     ->paginate($perPage);

            $bills = Bill::where(function ($query) {
                $query->where('approval', 2)
                    ->orWhereNull('approval');
                })
                ->where('delete_flag', 0)
                ->where('department_head', auth()->id())
                ->orderBy('id', 'desc')
                ->paginate($perPage);
        }

        return view('bill.awaiting-bills', compact('bills', 'perPage'));
    }


    // public function pendingBills(){
    //     if (!auth()->user()->can('View Bill')) {
    //         abort(403, 'Sorry !! You are not authorized!');
    //     }

    //     $bills = Bill::where('approval', null)
    //                  ->where('delete_flag', 0)
    //                  ->orderBy('id', 'desc')
    //                  ->get();

    //     return view('bill.pending-bills', compact('bills'));
    // }


    public function pendingBills(Request $request)
    {
        if (!auth()->user()->can('Pending Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        // Get the number of entries per page from the request, default to 25
        $perPage = $request->input('per_page', 50);

        // Fetch bills with pagination
        $bills = Bill::where('approval', null)
                     ->where('delete_flag', 0)
                     ->orderBy('id', 'desc')
                     ->paginate($perPage);

        return view('bill.pending-bills', compact('bills', 'perPage'));
    }


    // public function rejectedBills(){
    //     if (!auth()->user()->can('View Bill')) {
    //         abort(403, 'Sorry !! You are not authorized!');
    //     }

    //     $bills = Bill::where('approval', 0)->orderBy('id','desc')->get();
    //     return view('bill.rejected-bills', compact('bills'));
    // }

    public function rejectedBills(Request $request) {
        if (!auth()->user()->can('Rejected Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 50);

        $bills = Bill::where('approval', 0)->where('delete_flag', 0)->orderBy('id', 'desc')->paginate($perPage);

        return view('bill.rejected-bills', compact('bills', 'perPage'));
    }


    public function showRejectNote($id)
    {
        $notes = BillNote::where('bill_id',$id)->get();

        return view('bill.reject-bill-note', compact('notes'));
    }




    public function requestApproval($id){
        if (!auth()->user()->can('Edit Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        Bill::find($id)->update(['approval' => null]);

        return redirect()->route('bills.rejected')->with('success', 'Approval Requested successfully.');
    }


    public function changeStatus(Request $request, $id, $status)
    {
        $selectedBills = $request->input('selected_bills', []);
        $rejectNote = $request->input('reject_note');

        // If there are selected bills, handle them using the statusChange logic
        if (!empty($selectedBills)) {
            foreach ($selectedBills as $billId) {

                $bill = Bill::find($billId);
                $bill->approval = 0;
                $bill->save();

                $billNote              = new BillNote();
                $billNote->bill_id     = $billId;
                $billNote->note        = $rejectNote;
                $billNote->created_by  = auth()->id();
                $billNote->save();

                try {
                    $this->sendRejectEmail($bill, $rejectNote);
                } catch (\Exception $e) {
                }
            }
            return redirect()->route('bills.pending')->with('success', 'Bill Status Changed & Deleted Successfully.');
        }

        // Original changeStatus logic for handling a single bill
        $bill = Bill::find($id);

        if (!$bill) {
            return redirect()->route('bills.pending')->with('error', 'Bill not found.');
        }

        if ($status == 1) {
            $requiredFields = ['bill_subject_id', 'bill_group', 'status', 'vat_refund', 'order_sl_no'];

            foreach ($requiredFields as $field) {
                if (is_null($bill->$field)) {
                    return redirect()->route('bills.edit', $id)->with('error', 'Please, before approval more information is needed.');
                }
            }


            $bill->approval = 1;
            // $bill->invoice_no = $invoice_no;
            // $bill->sl_no = Bill::max('sl_no')+1;
            $bill->approved_by = auth()->id();
            $bill->save();

            return redirect()->route('bills.pending')->with('success', 'Bill Approved successfully.');

        } elseif ($status == 0) {
            $bill->approval = 0;
            $bill->save();

            $billNote              = new BillNote();
            $billNote->bill_id     = $bill->id;
            $billNote->note        = $rejectNote;
            $billNote->created_by  = auth()->id();
            $billNote->save();

            try {
                $this->sendRejectEmail($bill, $rejectNote);
            } catch (\Exception $e) {
            }

            return redirect()->route('bills.pending')->with('success', 'Bill Status Changed & Deleted Successfully.');
        }
    }


    private function sendRejectEmail(Bill $bill, $rejectNote)
    {

        $creator = User::findOrFail($bill->created_by);
        $email = $creator->email;

        $mailData = [

            'reject_note' => $rejectNote,
            'customer_id' => $bill->customer->customer_name,
            // 'bill_no' =>  $bill->id ? str_pad($bill->id, 5, '0', STR_PAD_LEFT),
            'bill_no' => isset($bill->id) ? str_pad($bill->id, 5, '0', STR_PAD_LEFT) : null,
            'particulars' => $bill->particulars,
            'convertion_totalamount' => $bill->convertion_total_amount,
            'start_date' => $bill->start_date,
            'end_date' => $bill->end_date,
            'bill_type' => $bill->bill_type,
            'vat_status' => $bill->vat_status == 1 ? 'Yes' : 'No',
            'vat_percent' =>  $bill->vat_percent,

        ];

        try {
            Mail::send('emails.rejectnote_bill', $mailData, function($message) use ($email) {
                $message->to($email)
                        ->subject('Reject Note Notification');
            });
        } catch (\Exception $e) {
            // Log the error message if needed
            // Log::error('Email could not be sent: ' . $e->getMessage());
        }

    }


    public function changeRejectedBatch(Request $request){

        if($request->status==1){
            if(($request->selected_bills)>0){
                foreach($request->selected_bills as $id){
                    Bill::find($id)->update(['approval' => null]);
                }
            }
            return redirect()->back()->with('success', 'Bill request for approval successfully.');
        }else{
            if(($request->selected_bills)>0){
                foreach($request->selected_bills as $id){
                    $bill = Bill::find($id);
                    if ($bill->workorder_file) {
                        Storage::disk('public')->delete('workorders/' . $bill->workorder_file);
                    }
                    if ($bill->instalation_file) {
                        Storage::disk('public')->delete('installation/' . $bill->instalation_file);
                    }
                    if ($bill->deliverychallan_file) {
                        Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
                    }

                    $bill->delete();
                    LogActivityHelper::addToLog('Bill Delete','Bill',$bill->id);
                }
            }
            return redirect()->back()->with('success', 'Bill deleted successfully.');
        }

    }

    public function changePendingBatch(Request $request)
    {
        $status = $request->status;
        $selectedBills = $request->selected_bills;

        if ($status == 1) { // approve
            if (count($selectedBills) > 0) {
                $approvedBills = [];
                $failedBills = [];

                foreach ($selectedBills as $id) {
                    $bill = Bill::find($id);

                    if (!$bill) {
                        continue;
                    }

                    // Check for required fields
                    $requiredFields = ['bill_subject_id', 'bill_group', 'status', 'vat_refund', 'order_sl_no'];
                    $allFieldsFilled = true;

                    foreach ($requiredFields as $field) {
                        if (is_null($bill->$field)) {
                            $allFieldsFilled = false;
                            break;
                        }
                    }

                    if ($allFieldsFilled) {

                        $bill->update([
                            'approval'      => 1,
                            'approved_by'   => auth()->id()
                        ]);
                        $approvedBills[] = $bill->id;

                    } else {
                        $failedBills[] = $bill->id;
                    }
                }

                $message = '';

                if (count($approvedBills) > 0) {
                    $message .= 'Bill(s) approved successfully.';
                    return redirect()->back()->with('success', $message);
                }

                if (count($failedBills) > 0) {
                    $message .= ' Some bills could not be approved due to missing required information.';
                    return redirect()->back()->with('error', $message);
                }

                // return redirect()->back()->with('success', $message);
            }
        } elseif ($status == 2) { // reject
            if (count($selectedBills) > 0) {
                foreach ($selectedBills as $id) {
                    Bill::find($id)->update(['approval' => 0]);
                }
                return redirect()->back()->with('success', 'Bill(s) Status Changed & Back Successfully');
            }
        } elseif ($status == 0) { // delete
            if (count($selectedBills) > 0) {
                foreach ($selectedBills as $id) {

                    $bill = Bill::find($id);
                    if ($bill->workorder_file) {
                        Storage::disk('public')->delete('workorders/' . $bill->workorder_file);
                    }
                    if ($bill->instalation_file) {
                        Storage::disk('public')->delete('installation/' . $bill->instalation_file);
                    }
                    if ($bill->deliverychallan_file) {
                        Storage::disk('public')->delete('deliverychallan/' . $bill->deliverychallan_file);
                    }

                    $bill->delete();
                    LogActivityHelper::addToLog('Bill Delete','Bill',$bill->id);

                }
                return redirect()->back()->with('success', 'Bill(s) Rejected successfully.');

            }
        }

        return redirect()->back()->with('error', 'No bills selected for processing.');
    }

    public function activeInactiveBills($id=null)
    {
        set_time_limit(0);

        $bills = [];
        $customerInfo = '';

        if($id!=''){
            $customerInfo = Customer::select('id', 'customer_name', 'address')->find($id);
            $bills = Bill::where('approval', '1')
                ->where('delete_flag', 0)
                ->where('customer_id', $id)->get();
        }

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('bill.active-inactive-bills', compact('bills', 'customers', 'customerInfo', 'id'));
    }

    public function billActiveStatus($id,$type)
    {
        if (Bill::where('id', $id)->update(['status' => $type])) {
            LogActivityHelper::addToLog("Bill $type",'Bill',$id);
            return $type === 'Active' ? 'Active' : 'Inactive';
        } else {
            return 404;
        }
    }

    public function changeStatusAwaiting($id, $status)
    {
        $bill = Bill::find($id);
        $bill->approval = ($status==200)? null:$status;
        $bill->awaiting_approved_by = auth()->id();
        $bill->save();

        if($status==200){
            // approved
            return redirect()->route('bills.awaiting')->with('success', 'Bill successfully approved.');
        }else{
            // reject
            return redirect()->route('bills.awaiting')->with('success', 'Bill successfully rejected.');
        }

    }

    public function changeStatusAwaitingBatch(Request $request)
    {
        $status = $request->status;
        $selectedBills = $request->selected_bills;

        if (count($selectedBills) > 0) {
            $approvedBills = [];
            $failedBills = [];

            foreach ($selectedBills as $id) {
                $bill = Bill::find($id);

                if (!$bill) {
                    continue;
                }

                $bill->update([
                    'approval'      => ($status==200)? null:$status,
                    'awaiting_approved_by'   => auth()->id()
                ]);
            }

        }

        if($status==200){
            // approved
            return redirect()->route('bills.awaiting')->with('success', 'Bill successfully approved.');
        }else{
            // reject
            return redirect()->route('bills.awaiting')->with('success', 'Bill successfully rejected.');
        }

    }

    public function getDeptUsers($deptId=null){

        $users = UserDept::leftJoin('users', 'user_depts.user_id', '=', 'users.id')
            ->where('user_depts.department_id', $deptId)
            // ->select('user_depts.*', 'users.name')
            ->select('users.id', 'users.name')
            ->get();
        $result = '<option value="">--Select Dept Head--</option>';
        foreach($users as $row){
            $result.='<option value="'.$row->id.'">'.$row->name.'</option>';
        }

        return $result;

    }

    public function billFixingFind($type=null,$keyid=null)
    {
        if (!auth()->user()->can('Edit Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $keyid = (int)$keyid;
        if($keyid<1){
            return view('bill.bill-fixing', compact('type','keyid'));
        }

        if($type=='serial'){
            $bill = Bill::where('bill_slno_x',$keyid)->first();
        }else{
            $bill = Bill::where('id',$keyid)->first();
        }

        $departments = [];
        $billSubjects = BillSubject::select('id', 'bill_subject','divName','hs_code')->orderBy('bill_subject')->get();
        $vatsetup = VatSetup::select('id', 'vat')->orderBy(DB::raw('CAST(vat AS SIGNED)'), 'asc')->get();
        $customers = Customer::where('status','Active')->orderBy('customer_name', 'asc')->get();

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        // Get department list from API
        try {
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        } catch (\Exception $e) {
            // Log::error($e->getMessage());
        }

        $autoAcNo = '';
        $binNo = '';
        if ($bill->customer_id && ($custInfo = Customer::find($bill->customer_id))) {
            $autoAcNo = $custInfo->autoAcNo;
            $binNo = $custInfo->bin_no;
        }

        return view('bill.bill-fixing', compact('type','keyid','bill', 'departments', 'billSubjects', 'customers', 'autoAcNo', 'vatsetup','binNo'))
            ->with('current_order_sl_no', $bill->order_sl_no);
    }

    public function billFixingStoreWorkBackup(Request $request, $bill_id=null){

        //exit;

        if (!auth()->user()->can('Edit Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $security_code = $request->security_code;
        if($security_code!=1200060){
            return redirect()->back()->with('error', 'Security code not matched.');
        }

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'qty' => 'required',
            'unit_price_vat' => 'required',
            'total_amount' => 'required',
            'bill_subject_id' => 'required',
            'currency' => 'required',
            'currency_rate' => 'required|numeric|min:1',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with('error', 'Missing required fields.');
        }


        if($request->actionMode=='create'){
            $updateMode = false;  // create new bill
        }else{
            $bill = Bill::find($bill_id);
            if(empty($bill)){
                return 'Bill not found';
            }else{
                $updateMode = true; // update this bill
            }
        }

        // Check customer available or not
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        }

        $dateFields = ['ref_date', 'installation_date', 'activation_date', 'start_date', 'end_date', 'increase_start_date'];
        $data = $request->all();
        $data['customer_id'] = $custPKID;

        if($request->vat_status=='0'){
            $data['vat_percent'] = null;
            $data['vat_refund'] = 0;
        }else{
            $data['vat_refund'] = 1;
        }

        $data['conv_unit_total'] = round(($request->unit_total * $request->currency_rate),2);
        $data['convertion_total_amount'] = round(($request->total_amount * $request->currency_rate),2);
        $data['convertion_total_amount_vat'] = round(($request->total_amount_vat * $request->currency_rate),2);
        $data['conv_vat_amount'] = round(($request->vat_amount * $request->currency_rate),2);

        foreach ($dateFields as $field) {
            if ($request->filled($field)) {
                $data[$field] = $this->normalizeDate($request->$field);
            }
        }

        $data['approval'] = 1;
        $data['approved_by'] = 1;
        $data['awaiting_approved_by'] = 1;
        $data['department_head'] = 1;
        $data['bill_editor_id'] = 1;
        $data['bill_edited'] = 1;
        $data['delete_flag'] = 0;

        if(!$updateMode){ // create
            // Attention Person
            $attention_id = trim($fndCustomer->attention_person);
            if($attention_id!='' && !empty($attenInfo=AttentionPerson::find((int)$attention_id))){
                $data['attention_id'] = $attenInfo->id;
                $data['attention_person'] = $attenInfo->attention_name;
            }
        }

        //$result[] = $bill;
        $result[] = $data;

        // return $result;

        if(!$updateMode){
            // return 'create';
            $data['bill_old'] = 1;
            $billInfo = Bill::create($data + ['created_by' => auth()->id()]);
            $tblName = $billInfo->getTable();
            if($billInfo){
                DB::table($tblName)->where('id', $billInfo->id)->update(['bill_old' => 1,'payment_type' => 'A']);
                return redirect()->route('billFixingFind', ['type' => 'id', 'id' => $billInfo->id])
                ->with('success', 'Bill created successfully.');
            }else{
                return redirect()->route('billFixingFind')
                ->with('error', 'Bill not created.');
            }
        }else{
            // return 'update';
            $billInfo = $bill->update($data + ['updated_by' => auth()->id()]);
            if($billInfo){
                return redirect()->route('billFixingFind', ['type' => 'id', 'id' => $bill->id])
                ->with('success', 'Bill updated successfully.');
            }else{
                return redirect()->route('billFixingFind', ['type' => 'id', 'id' => $billInfo->id])
                ->with('error', 'Bill not updated.');
            }
        }


    }

    public function billFixingStore(Request $request, $bill_id=null){

        //exit;

        if (!auth()->user()->can('Edit Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $security_code = $request->security_code;
        if($security_code!=1200060){
            return redirect()->back()->with('error', 'Security code not matched.');
        }

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'qty' => 'required',
            'unit_price_vat' => 'required',
            'total_amount' => 'required',
            'bill_subject_id' => 'required',
            'currency' => 'required',
            'currency_rate' => 'required|numeric|min:1',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with('error', 'Missing required fields.');
        }


        if($request->actionMode=='create'){
            $updateMode = false;  // create new bill
        }else{
            $bill = Bill::find($bill_id);
            if(empty($bill)){
                return 'Bill not found';
            }else{
                $updateMode = true; // update this bill
            }
        }

        // Check customer available or not
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        }

        $dateFields = ['ref_date', 'installation_date', 'activation_date', 'start_date', 'end_date', 'increase_start_date'];
        $data = $request->all();
        $data['customer_id'] = $custPKID;

        if($request->vat_status=='0'){
            $data['vat_percent'] = null;
            $data['vat_refund'] = 0;
        }else{
            $data['vat_refund'] = 1;
        }

        $data['conv_unit_total'] = round(($request->unit_total * $request->currency_rate),2);
        $data['convertion_total_amount'] = round(($request->total_amount * $request->currency_rate),2);
        $data['convertion_total_amount_vat'] = round(($request->total_amount_vat * $request->currency_rate),2);
        $data['conv_vat_amount'] = round(($request->vat_amount * $request->currency_rate),2);

        foreach ($dateFields as $field) {
            if ($request->filled($field)) {
                $data[$field] = $this->normalizeDate($request->$field);
            }
        }

        // $data['approval'] = 1;
        // $data['approved_by'] = 1;
        // $data['awaiting_approved_by'] = 1;
        // $data['department_head'] = 1;
        // $data['bill_editor_id'] = 1;
        // $data['bill_edited'] = 1;
        // $data['delete_flag'] = 0;

        if(!$updateMode){ // create
            // Attention Person
            $attention_id = trim($fndCustomer->attention_person);
            if($attention_id!='' && !empty($attenInfo=AttentionPerson::find((int)$attention_id))){
                $data['attention_id'] = $attenInfo->id;
                $data['attention_person'] = $attenInfo->attention_name;
            }
        }

        //$result[] = $bill;
        $result[] = $data;

        // return $result;

        if(!$updateMode){
            // return 'create';
            $data['bill_old'] = 1;
            $billInfo = Bill::create($data + ['created_by' => auth()->id()]);
            $tblName = $billInfo->getTable();
            if($billInfo){
                DB::table($tblName)->where('id', $billInfo->id)->update(['bill_old' => 1,'payment_type' => 'A']);
                return redirect()->route('billFixingFind', ['type' => 'id', 'id' => $billInfo->id])
                ->with('success', 'Bill created successfully.');
            }else{
                return redirect()->route('billFixingFind')
                ->with('error', 'Bill not created.');
            }
        }else{
            // return 'update';


            // Bill Revision Store - Start
            $watchedColumns = [
                'bill_subject_id', 'particulars', 'bill_group', 'qty', 'unit_price_vat',
                'vat_status', 'vat_percent', 'vat_type', 'ref_no', 'ref_date',
                'installation_date', 'activation_date', 'start_date', 'end_date',
                'increase_start_date', 'yearly_increase_percent', 'increase_period',
                'bill_type', 'status', 'department_id'
            ];

            // Compare fields — check if any have changed
            $hasChanges = false;
            foreach ($watchedColumns as $column) {
                $oldValue = $bill->$column ?? '';
                $newValue = $request->$column ?? '';
                if ($oldValue != $newValue) {
                    $hasChanges = true;
                    break;
                }
            }

            // If there are changes, insert old version (entire old row) into billing_bill_revisions
            if ($hasChanges) {
                $revisionData = $bill->toArray(); // full old record

                // Add revision tracking info
                $maxRevision = BillRevision::where('bill_id', $bill->id)->max('revision_no');
                $revisionData['revision_no'] = $maxRevision ? $maxRevision + 1 : 1;

                $revisionData['bill_id'] = $bill->id;
                $revisionData['revised_by'] = auth()->id();
                $revisionData['created_at'] = now();
                $revisionData['updated_at'] = now();

                BillRevision::create($revisionData);
            }
            // Bill Revision Store - End


            // Now update with new values
            $billInfo = $bill->update($data + ['updated_by' => auth()->id()]);

            if($billInfo){
                return redirect()->route('billFixingFind', ['type' => 'id', 'id' => $bill->id])
                ->with('success', 'Bill updated successfully.');
            }else{
                return redirect()->route('billFixingFind', ['type' => 'id', 'id' => $billInfo->id])
                ->with('error', 'Bill not updated.');
            }
        }


    }

    public function endingBills($month_no = null, $customer_id = null)
    {
        set_time_limit(0);

        if($month_no==''){
            $month_no = 1;
        }

        $bills = [];

        $query = Bill::query();

        // Convert end_date (d/m/Y) into DATE for filtering
        // MySQL: STR_TO_DATE(end_date, '%d/%m/%Y')
        $currentMonth = now()->month;
        $currentYear  = now()->year;

        if ($month_no == 1) {
            // Current month
            $targetMonth = $currentMonth;
            $targetYear  = $currentYear;
        } elseif ($month_no == 2) {
            // Next month
            $targetMonth = now()->addMonth()->month;
            $targetYear  = now()->addMonth()->year;
        } elseif ($month_no == 3) {
            // Next +1 month
            $targetMonth = now()->addMonths(2)->month;
            $targetYear  = now()->addMonths(2)->year;
        } elseif ($month_no == 4) {
            // Next +1 month
            $targetMonth = now()->addMonths(3)->month;
            $targetYear  = now()->addMonths(3)->year;
        } else {
            return back()->with('error', 'Invalid option selected.');
        }

        // Apply date filter using MySQL date conversion
        $query->whereRaw("MONTH(STR_TO_DATE(end_date, '%d/%m/%Y')) = ?", [$targetMonth])
            ->whereRaw("YEAR(STR_TO_DATE(end_date, '%d/%m/%Y')) = ?", [$targetYear]);

        $query->where('approval', '1');
        $query->where('delete_flag', 0);

        // Optional customer filter
        if (!empty($customer_id)) {
            $query->where('customer_id', $customer_id);
        }

        $bills = $query->get();

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('bill.ending-bills', compact('bills', 'customers', 'month_no', 'customer_id'));
    }


    private function normalizeDate($date)
    {
        $formats = [
            'd/m/y',
            'd/m/Y',
            'd-M-Y',
            'd-M-y'
        ];

        foreach ($formats as $format) {
            try {
                $parsedDate = \Carbon\Carbon::createFromFormat($format, $date);
                return $parsedDate->format('d/m/Y');
            } catch (\Exception $e) {
            }
        }

        return $date;
    }

}
