<?php

namespace App\Http\Controllers;

use App\Http\Controllers\ReportController;
use Illuminate\Http\Request;
use App\Models\Bill;
use App\Models\BillProcess;
use App\Models\BillSubject;
use App\Models\Customer;
use App\Models\CustomerLedger;
use App\Models\CollectionChd;
use App\Models\GlHead;
use App\Models\Settings;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use App\Mail\ProcessedBill;
use Illuminate\Support\Facades\Log;
use Barryvdh\DomPDF\Facade\Pdf;
use DateTime;

class BillProcessController extends Controller
{
    public function __construct()
    {
        set_time_limit(0);
        ini_set('memory_limit', '3000M');
        ini_set('post_max_size', '10M');
        ini_set('max_input_vars', '3000');
    }

    public function index()
    {
        if (!auth()->user()->can('Process Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        return view('bill-process.index');
    }

    public function invoicesOKBackupForOrdering(Request $request)
    {
        if (!auth()->user()->can('Show Invoice')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 50);

        $query = BillProcess::query();
        $query->where('is_print', 0);
        $query->where('delete_flag', 0);

        // Apply filters
        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('bill_type')) {
            $query->where('bill_type', $request->bill_type);
        }

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('invoice_no')) {
            $query->where('invoice_no', $request->invoice_no);
        }

        $groupedBills = $query->select(
            'customer_id',
            'bill_subject_desc',
            'invoice_no',
            'bill_group',
            DB::raw("MAX(invoice_date) as invoice_date"),
            DB::raw('GROUP_CONCAT(id) as bill_ids'),
            DB::raw('SUM(total_amount_vat) as total_vat_amount')
        )
        ->groupBy('customer_id', 'bill_subject_desc', 'invoice_no','bill_group')
        ->orderBy(DB::raw('MAX(id)'), 'desc')
        ->paginate($perPage)
        ->appends($request->except('page'));


        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('invoice.index', compact('groupedBills', 'perPage','customers'));

    }


    public function invoices(Request $request)
    {
        if (!auth()->user()->can('Show Invoice')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 50);

        $query = BillProcess::query();
        $query->where('is_print', 0);
        $query->where('delete_flag', 0);

        // Apply filters
        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('bill_type')) {
            $query->where('bill_type', $request->bill_type);
        }

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('invoice_no')) {
            $query->where('invoice_no', $request->invoice_no);
        }

        $groupedBills = $query->select(
            'customer_id',
            'bill_subject_desc',
            'invoice_no',
            'bill_group',
            DB::raw("MAX(invoice_date) as invoice_date"),
            DB::raw("MAX(id) as max_id"),
            DB::raw('GROUP_CONCAT(id) as bill_ids'),
            DB::raw('SUM(total_amount_vat) as total_vat_amount')
        )
        ->groupBy('customer_id', 'bill_subject_desc', 'invoice_no', 'bill_group')
        ->orderByRaw("STR_TO_DATE(MAX(invoice_date), '%d/%m/%Y') DESC")
        ->orderByRaw("MAX(id) DESC")
        ->paginate($perPage)
        ->appends($request->except('page'));


        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('invoice.index', compact('groupedBills', 'perPage','customers'));

    }



    // Single invoice preview
    public function showGrouped(Request $request)
    {
        // $billIds = $request->input('bill_ids');

        // if (!$billIds) {
        //     abort(404, 'No bill IDs provided.');
        // }

        // $billIdsArray = explode(',', $billIds);
        // $invoices = BillProcess::whereIn('id', $billIdsArray)->get();

        $invNO = $request->input('invNO');
        $invNO = str_replace("_","/",$invNO);
        $invoices = BillProcess::where('invoice_no', $invNO)->get();

        $customerIds = $invoices->pluck('customer_id')->unique();
        $customers = \App\Models\Customer::whereIn('id', $customerIds)->get()->keyBy('id');

        $customerNames = $invoices->map(function ($invoice) {
            return $invoice->customer ? $invoice->customer->customer_name : '';
        })->unique();

        $firstCustomerName = $customerNames->first();
        $firstCustomerBinNo = $invoices[0]->customer->bin_no ?? '';

        // Retrieve the first invoice's number and date for display
        $firstInvoice = $invoices->first();
        $invoiceNo = $firstInvoice ? $firstInvoice->invoice_no : '';
        $invoiceDate = $firstInvoice ? $firstInvoice->invoice_date : '';
        $currency = $firstInvoice ? $firstInvoice->currency : '';
        $attentionPerson = $firstInvoice ? $firstInvoice->attention_person : '';
        $firstBillSubject = $firstInvoice ? $firstInvoice->bill_subject_desc : '';

        // Retrieve the address for the first customer
        $firstCustomerId = $invoices->first()->customer_id ?? '';
        $address = $firstCustomerId ? ($customers->get($firstCustomerId)->address ?? '') : '';

        return view('invoice.grouped-invoice', [
            'invoices' => $invoices,
            'firstCustomerName' => $firstCustomerName,
            'firstCustomerBinNo' => $firstCustomerBinNo,
            'firstBillSubject' => $firstBillSubject,
            'invoiceNo' => $invoiceNo,
            'currency' => $currency,
            'attentionPerson' => $attentionPerson,
            'invoiceDate' => $invoiceDate,
            'address' => $address
        ]);
    }


    // Bill Pre Invoice Preview
    public function preInvoice($id)
    {
        $bill = Bill::find($id);
        $customer = Customer::find($bill->customer_id);
        $firstCustomerName = $customer->customer_name;
        $firstCustomerBinNo = $customer->bin_no;
        $address = $customer->address;
        $billDate = date('d/m/Y', strtotime($bill->created_at));
        $firstBillSubject = BillSubject::find($bill->bill_subject_id);

        return view('invoice.pre-invoice', [
            'bill' => $bill,
            'firstCustomerName' => $firstCustomerName,
            'firstCustomerBinNo' => $firstCustomerBinNo,
            'firstBillSubject' => $firstBillSubject,
            'invoiceNo' => str_pad($bill->id, 5, '0', STR_PAD_LEFT),
            'invoiceDate' => $billDate,
            'address' => $address,
        ]);
    }

    // Pre invoice PDF generate
    public function invoicePdfPre(Request $request,$type){

        $data['INVOICE_NO'] = [$request->BILL_NO[0]];
        $billNo = str_pad($request->BILL_NO[0], 5, '0', STR_PAD_LEFT);
        $name = "Pre_invoice_(bill_no)_".$billNo.".pdf";

        $pdf = Pdf::loadView('invoice.invoice-pdf-pre', $data);
        $pdf->setPaper('A4', 'portrait');
        // return $pdf->stream($name, ['Attachment'=>false]);
        return $pdf->download($name); // Force download

    }


    // Print preview invoice after check & click print invoice
    public function invoicesAllGroupbywise(Request $request)
    {
        $selectedBills = json_decode($request->input('selected_bills'), true);
        $showAll = $request->has('show_all') && $request->input('show_all') == 'true';

        if (!empty($selectedBills) || $showAll) {
            if ($showAll) {

                // not coming
                $invoiceall = BillProcess::groupBy('invoice_no')
                    ->select('invoice_no',
                             DB::raw('MIN(customer_id) as customer_id'),
                             DB::raw('MIN(invoice_date) as invoice_date'),
                             DB::raw('MIN(bill_subject_id) as bill_subject_id'))
                    ->orderBy('id','asc')
                    ->get();
            } else {

                // goes this
                // $invoiceall = BillProcess::whereIn('invoice_no', $selectedBills)
                //     ->groupBy('invoice_no')
                //     ->select('invoice_no',
                //              DB::raw('MIN(customer_id) as customer_id'),
                //              DB::raw('MIN(invoice_date) as invoice_date'),
                //              DB::raw('MIN(bill_subject_id) as bill_subject_id'))
                //     ->orderBy('id','asc')
                //     ->get();

                // come in here
                $invoiceall = BillProcess::whereIn('invoice_no', $selectedBills)
                ->groupBy('invoice_no') // Group only by 'invoice_no'
                ->select(
                    'invoice_no',
                    DB::raw('MIN(customer_id) as customer_id'), // Aggregate customer_id
                    DB::raw('MIN(invoice_date) as invoice_date'), // Aggregate invoice_date
                    DB::raw('MIN(bill_subject_desc) as bill_subject_desc'), // Aggregate bill_subject_id
                    DB::raw('MIN(id) as id') // Use MIN or MAX for id
                )
                // ->orderBy('id', 'asc') // Sort by id
                ->orderByRaw("STR_TO_DATE(MIN(invoice_date), '%d/%m/%Y') ASC") // Correct order
                ->get();
            }
        } else {

            // return redirect()->route('all.invoices.groupbywise')->with('error', 'No invoices selected.');
            return redirect()->route('invoices.index')->with('error', 'No invoices selected.');
        }

        $invoiceDetails = $invoiceall->map(function($invoice) {
            $customer = Customer::find($invoice->customer_id);

            return [
                'invoice_no' => $invoice->invoice_no,
                'invoice_date' => $invoice->invoice_date,
                'customer_name' => $customer->customer_name ?? '',
                'customer_bin_no' => $customer->bin_no ?? '',
                'address' => $customer->address ?? '',
                'bill_subject' => $invoice->bill_subject_desc
            ];
        });

        return view('invoice.groupall-invoice', compact('invoiceDetails'));
    }

    // Invoice PDF generate
    public function invoicePdf(Request $request,$type){

        if($type=='list'){
            $data['INVOICE_NO'] = $request->INVOICE_NO;
            $name = "Invoice_list_".date('Y-m-d').".pdf";
        }else if($type=='single'){
            $data['INVOICE_NO'] = [$request->INVOICE_NO[0]];
            $info = str_replace("/", "-", $request->INVOICE_NO[0]);
            $name = "Invoice_no_".$info.".pdf";
        }

        $pdf = Pdf::loadView('invoice.invoice-pdf', $data);
        $pdf->setPaper('A4', 'portrait');
        // return $pdf->stream($name, ['Attachment'=>false]);
        return $pdf->download($name); // Force download

    }


    public function invoicesAllGroupbywiseshow(Request $request)
    {
        $query = BillProcess::query();
        $query->where('is_print', 0);
        $query->where('delete_flag', 0);

        // Apply filters if present
        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('bill_type')) {
            $query->where('bill_type', $request->bill_type);
        }

        // if ($request->filled('payment_status')) {
        //     $query->where('payment_status', $request->payment_status);
        // }

        // Group by logic
        $invoiceall = $query->groupBy('invoice_no')
            ->select('invoice_no',
                     DB::raw('MIN(customer_id) as customer_id'),
                     DB::raw('MIN(invoice_date) as invoice_date'),
                     DB::raw('MIN(bill_subject_id) as bill_subject_id'))
            ->get();

        $invoiceDetails = $invoiceall->map(function($invoice) {
            $customer = Customer::find($invoice->customer_id);
            $billSubject = BillSubject::find($invoice->bill_subject_id);

            return [
                'invoice_no' => $invoice->invoice_no,
                'invoice_date' => $invoice->invoice_date,
                'customer_name' => $customer->customer_name ?? '',
                'address' => $customer->address ?? '',
                'bill_subject' => $billSubject->bill_subject ?? ''
            ];
        });

        return view('invoice.groupall-invoice', compact('invoiceDetails'));
    }



    // Duplicate Invoice
    public function duplicateInvoice(Request $request)
    {

        $perPage = $request->input('per_page', 50);

        $query = BillProcess::query();
        $query->where('is_print', 1);
        $query->where('delete_flag', 0);

        // Apply filters
        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(invoice_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('bill_type')) {
            $query->where('bill_type', $request->bill_type);
        }

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('invoice_no')) {
            $query->where('invoice_no', $request->invoice_no);
        }

        $bills = [];

        $groupedBills = $query->select(
            'customer_id',
            'bill_subject_desc',
            'invoice_no',
            'bill_group',
            DB::raw("MAX(invoice_date) as invoice_date"),
            DB::raw("MAX(id) as max_id"),
            DB::raw('GROUP_CONCAT(id) as bill_ids'),
            DB::raw('SUM(total_amount_vat) as total_vat_amount')
        )
        ->groupBy('customer_id', 'bill_subject_desc', 'invoice_no','bill_group')
        ->orderByRaw("STR_TO_DATE(MAX(invoice_date), '%d/%m/%Y') DESC")
        ->orderByRaw("MAX(id) DESC")
        ->paginate($perPage)
        ->appends($request->except('page'));

        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('invoice.duplicate-group-invoice', compact('groupedBills', 'perPage', 'bills','customers'));

    }


    // Duplicate invoice preview
    public function duplicateInvoiceView(Request $request)
    {
        // $billIds = $request->input('bill_ids');

        // if (!$billIds) {
        //     abort(404, 'No bill IDs provided.');
        // }

        // $billIdsArray = explode(',', $billIds);
        // $invoices = BillProcess::whereIn('id', $billIdsArray)->get();

        $invNO = $request->input('invNO');
        $invNO = str_replace("_","/",$invNO);
        $invoices = BillProcess::where('invoice_no', $invNO)->get();

        $customerIds = $invoices->pluck('customer_id')->unique();
        $customers = \App\Models\Customer::whereIn('id', $customerIds)->get()->keyBy('id');

        $customerNames = $invoices->map(function ($invoice) {
            return $invoice->customer ? $invoice->customer->customer_name : '';
        })->unique();

        $firstCustomerName = $customerNames->first();
        $firstCustomerBinNo = $invoices[0]->customer->bin_no ?? '';

        // Retrieve the first invoice's number and date for display
        $firstInvoice = $invoices->first();
        $invoiceNo = $firstInvoice ? $firstInvoice->invoice_no : '';
        $invoiceDate = $firstInvoice ? $firstInvoice->invoice_date : '';
        $currency = $firstInvoice ? $firstInvoice->currency : '';
        $attentionPerson = $firstInvoice ? $firstInvoice->attention_person : '';
        $firstBillSubject = $firstInvoice ? $firstInvoice->bill_subject_desc : '';

        // Retrieve the address for the first customer
        $firstCustomerId = $invoices->first()->customer_id ?? '';
        $address = $firstCustomerId ? ($customers->get($firstCustomerId)->address ?? '') : '';

        return view('invoice.duplicate-group-invoice-view', [
            'invoices' => $invoices,
            'firstCustomerName' => $firstCustomerName,
            'firstCustomerBinNo' => $firstCustomerBinNo,
            'firstBillSubject' => $firstBillSubject,
            'attentionPerson' => $attentionPerson,
            'invoiceNo' => $invoiceNo,
            'currency' => $currency,
            'invoiceDate' => $invoiceDate,
            'address' => $address
        ]);
    }

    // Duplicte invoice PDF generate
    public function invoicePdfDuplicate(Request $request,$type){

        if($type=='list'){
            $data['INVOICE_NO'] = $request->INVOICE_NO;
            $name = "Invoice_list_(duplicate)_".date('Y-m-d').".pdf";
        }else if($type=='single'){
            $data['INVOICE_NO'] = [$request->INVOICE_NO[0]];
            $info = str_replace("/", "-", $request->INVOICE_NO[0]);
            $name = "Invoice_no_(duplicate)_".$info.".pdf";
        }

        $pdf = Pdf::loadView('invoice.invoice-pdf-duplicate', $data);
        $pdf->setPaper('A4', 'portrait');
        // return $pdf->stream($name, ['Attachment'=>false]);
        return $pdf->download($name); // Force download

    }

    public function setPrintFlag($invoiceNo){
        $new_string = str_replace("_", "/", $invoiceNo);
        $invInfo = BillProcess::where('invoice_no',$new_string)->get();
        if($invInfo){
            foreach($invInfo as $row){
                $row->is_print = 1;
                $row->printed_by = auth()->id();
                $row->printed_date = date('Y-m-d');
                $row->save();
            }
        }
    }

    public function setPrintCount($invoiceNo){
        $new_string = str_replace("_", "/", $invoiceNo);
        $invInfo = BillProcess::where('invoice_no',$new_string)->get();
        if($invInfo){
            foreach($invInfo as $row){
                $row->duplicate_print_count = BillProcess::max('duplicate_print_count')+1;
                $row->save();
            }
        }
    }


    public function showInvoice($id)
    {
        if (!auth()->user()->can('Show Invoice')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $invoice=BillProcess::find($id);
        return view('invoice.invoice', compact('invoice'));
    }




    public function allInvoices(Request $request)
    {
        if (!auth()->user()->can('Show Invoice')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $billsData = json_decode($request->bills, true);

        $billsData = $billsData['data'] ?? $billsData;

        if (empty($billsData)) {
            abort(400, 'No bills data found.');
        }

        $invoices = array_map(function ($billData) {
            return (new BillProcess())->newInstance($billData, true);
        }, $billsData);

        return view('invoice.all-invoices', compact('invoices'));
    }

    // find bill list for processing
    // This is working
    public function findBills(Request $request)
    {
        if (!auth()->user()->can('Process Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $increaseSettings = Settings::where('name','increase_check_before_process')->first() ?? '';
        if($increaseSettings->value=='yes'){
            $lastDay = date("d-m-Y", strtotime("last day of {$request->year}-{$request->month}"));
            $report = new ReportController;
            $count = $report->billIncreaseWarningCount($lastDay);
            if((int)$count > 0){
                return back()->with('error', 'Some bills have increasement, please fix it.');
            }
        }

        $billfilters = $request;
        $bills = [];
        if ($request->bill_type == 'One Time') {
            $requestedMonth = "{$request->month}/{$request->year}";
            $startOfMonth = Carbon::createFromFormat('F/Y', $requestedMonth)->startOfMonth()->format('Y-m-d');
            $endOfMonth = Carbon::createFromFormat('F/Y', $requestedMonth)->endOfMonth()->format('Y-m-d');
            $data = Bill::select('id', 'bill_type', 'status', 'approval', 'vat_refund', 'start_date', 'end_date',
                'customer_id','bill_group','qty','unit_price_vat','currency','currency_rate','convertion_total_amount',
                'conv_vat_amount','convertion_total_amount_vat','bill_type','bill_subject_id','billDivName','bill_old')
                ->where('bill_type', 'One Time')
                ->where('status', 'Active')
                ->where('approval', '1')
                ->where('vat_refund', $request->vat_refund)
                ->where(function ($query) use ($startOfMonth, $endOfMonth) {
                    $query->whereRaw(
                        "STR_TO_DATE(start_date, '%d/%m/%Y') BETWEEN ? AND ?
                        OR STR_TO_DATE(end_date, '%d/%m/%Y') BETWEEN ? AND ?",
                        [$startOfMonth, $endOfMonth, $startOfMonth, $endOfMonth]
                    );
                })
                ->with(['billProcesses:id,bill_id,month,year'])
                ->with(['billSubject:id,bill_subject'])
                ->with(['customer:id,customer_name,autoAcNo'])
                ->get();
            foreach ($data as $value) {
                if ($value->billProcesses->isEmpty()) {
                    $bills[] = $value;
                }
            }
        } else {
            $recurringTypes = ['Monthly', 'Quarterly', 'Half Yearly', 'Yearly'];
            foreach ($recurringTypes as $type) {
                $interval = match ($type) {
                    'Monthly' => 1,
                    'Quarterly' => 3,
                    'Half Yearly' => 6,
                    'Yearly' => 12,
                    default => 1,
                };
                $data = Bill::select('id', 'bill_type', 'status', 'approval', 'vat_refund', 'start_date', 'end_date',
                    'customer_id','bill_group','qty','unit_price_vat','currency','currency_rate','convertion_total_amount',
                    'conv_vat_amount','convertion_total_amount_vat','bill_type','bill_subject_id','billDivName','bill_old')
                    ->where('bill_type', $type)
                    ->where('status', 'Active')
                    ->where('approval', '1')
                    ->where('vat_refund', $request->vat_refund)
                    ->where(function ($query) use ($request) {
                        $startOfMonth = date("Y-m-d", strtotime("last day of {$request->year}-{$request->month}"));
                        $query->where(DB::raw("STR_TO_DATE(start_date, '%d/%m/%Y')"), '<=', $startOfMonth);
                    })
                    ->with(['billProcesses:id,bill_id,month,year'])
                    ->with(['billSubject:id,bill_subject'])
                    ->with(['customer:id,customer_name,autoAcNo'])
                    ->get();
                foreach ($data as $value) {
                    $start_date = \DateTime::createFromFormat('d/m/Y', $value->start_date);
                    $end_date = \DateTime::createFromFormat('d/m/Y', $value->end_date);
                    if (!$start_date || !$end_date) {
                        continue;
                    }
                    $current_date = clone $start_date;
                    $billMatched = false;
                    while ($current_date <= $end_date) {
                        if ($current_date->format('F') === $request->month && $current_date->format('Y') == $request->year) {
                            $billMatched = true;
                            break;
                        }
                        $current_date->modify("+{$interval} months");
                    }
                    if ($billMatched) {
                        $processedBill = $value->billProcesses
                            ->where('month', $request->month)
                            ->where('year', $request->year);
                        if ($processedBill->isEmpty()) {
                            $bills[] = $value;
                        }
                    }
                }
            }
        }

        return view('bill-process.billProcess', compact('bills', 'billfilters'));
    }



    public function store(Request $request)
    {
        if (!auth()->user()->can('Process Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $voucherSettings = Settings::where('name','bill_process_account_hit')->first();

        $startTime = microtime(true);

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");
        $voucherError = false;

        if($token==''){
            return redirect()->route('bill-process.index')->with('error', 'API Connection Time Out.');
        }

        // TDS/TAX Amount
        $getVatHead = GlHead::select('gl_code')->where('gl_subject',strtoupper('VAT'))->first();
        if($getVatHead){
            $glCodeVat = $getVatHead->gl_code;
            try{
                $response4 = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                    'groupCode' => $glCodeVat,
                    'companyNo' => $company_number,
                ]);

                if ($response4->successful()) {
                    $responseData4 = $response4->json();
                    if ($responseData4 && isset($responseData4['accHeadDetailsInfo'])) {

                        $vatAutoAcNoAPI = $responseData4['accHeadDetailsInfo']['autoAcNo'];
                        $vatDivCodeAPI = $responseData4['accHeadDetailsInfo']['divCode'];
                        $vatDivNameAPI = $responseData4['accHeadDetailsInfo']['divName'];
                    }
                }
            }catch (\Exception $e) {
                if($e->getMessage()){
                    return redirect()->route('bill-process.index')->with('error', 'API Connection Time Out.');
                }
            }
        }

        // Fetch necessary inputs
        $selectedBillsIds = array_map('intval', $request->selected_bills);
        $remarks = trim($request->remarks);
        $invoice_date = $request->invoice_date;
        $entryRefund = $request->vat_refund;
        $entryDate = date('Y-m-d');
        $month = $request->month;
        $year = $request->year;

        // Get process SL number
        $processSLNo = BillProcess::max('process_no') + 1;
        $processSLFormat = sprintf('%08d%03d', date('Ymd'), $processSLNo);
        $processUUID = mt_rand(111111, 999999);

        // Fetch bills and group them
        $bills = Bill::with([
            'billSubject:id,divCode,divName,autoAcNo,bill_subject',
            'customer:id,customer_name,autoAcNo',
        ])
        ->whereIn('id', $selectedBillsIds)
        ->orderBy('customer_id')
        ->orderBy('bill_subject_id')
        ->orderBy('bill_group')
        ->orderBy('currency')
        ->orderBy('department_id')
        ->orderBy('id')
        ->get();

        $billsGroup = $bills->groupBy(function ($bill) {
            return $bill->customer_id . '-' . $bill->bill_subject_id . '-' . $bill->bill_group . '-' . $bill->currency . '-' . $bill->department_id;
        });

        // Get the last record
        $lastRecord = BillProcess::select(['invoice_no', 'vat_refund'])
            ->where('invoice_no', '!=', '')
            ->whereNotNull('invoice_no')
            ->orderByDesc('id')
            ->first();

        // $entryDate = '2025-07-01';
        $currentYear = date('y', strtotime($entryDate));
        $currentMonth = date('m', strtotime($entryDate));
        $lastInvNo = $lastRecord ? $lastRecord->invoice_no:'';
        $lastRefund = $lastRecord ? $lastRecord->vat_refund:'';
        $newInvoiceNo = $this->generateInvoiceNumber($lastInvNo,$lastRefund,$entryDate,$entryRefund);
        if($entryRefund=='1'){
            $serial = (int) substr($newInvoiceNo, 4);
        }else{
            $serial = (int) substr($newInvoiceNo, 0,8);
        }

        $billsArray = [];
        $x = 1;
        foreach ($billsGroup as $groupKey => $group) {
            $inv_no = $entryRefund == '1'
                ? sprintf('%02d%02d%08d', $currentYear, $currentMonth, $serial)
                : sprintf('%08d%02d%02d', $serial, $currentMonth, $currentYear);

            foreach ($group as $bill) {
                $eachRowUUID = mt_rand(11111111, 99999999);
                $billsArray[] = [
                    'bill_id' => $bill->id,
                    'customer_id' => $bill->customer_id,
                    'bill_subject_id' => $bill->bill_subject_id,
                    'bill_subject_desc' => ($bill->billSubject) ? $bill->billSubject->bill_subject . ' ' . $remarks : null,
                    'particulars' => $bill->particulars,
                    'currency' => $bill->currency,
                    'currency_rate' => $bill->currency_rate,
                    'qty' => $bill->qty,
                    'unit_price_vat' => $bill->unit_price_vat,
                    'unit_total' => $bill->unit_total,
                    'conv_unit_total' => $bill->conv_unit_total,
                    'total_amount' => $bill->total_amount,
                    'convertion_total_amount' => $bill->convertion_total_amount,
                    'vat_type' => $bill->vat_type,
                    'vat_status' => $bill->vat_status,
                    'vat_percent' => $bill->vat_percent,
                    'vat_amount' => $bill->vat_amount,
                    'conv_vat_amount' => $bill->conv_vat_amount,
                    'total_amount_vat' => $bill->total_amount_vat,
                    'convertion_total_amount_vat' => $bill->convertion_total_amount_vat,
                    'due' => $bill->total_amount_vat,
                    'ref_no' => $bill->ref_no,
                    'ref_date' => $bill->ref_date,
                    'installation_date' => $bill->installation_date,
                    'activation_date' => $bill->activation_date,
                    'start_date' => $bill->start_date,
                    'end_date' => $bill->end_date,
                    'attention_id' => $bill->attention_id,
                    'attention_person' => $bill->attention_person,
                    'department_id' => $bill->department_id,
                    'billDivName' => $bill->billDivName,
                    'bill_group' => $bill->bill_group,
                    'bill_type' => $bill->bill_type,
                    'vat_refund' => $entryRefund ? '1' : '0',
                    'yearly_increase_percent' => $bill->yearly_increase_percent,
                    'increase_start_date' => $bill->increase_start_date,
                    'payment_status' => 'Unpaid',
                    'hs_code' => $bill->hs_code,
                    'order_sl_no' => $bill->order_sl_no,
                    'month' => $month,
                    'year' => $year,
                    'invoice_no' => $inv_no,
                    'process_no' => $processSLNo,
                    'process_no_format' => $processSLFormat,
                    'process_uuid' => $processUUID,
                    // 'sl_no' => ($inv_no==250100000070)? '--':null,
                    'invoice_date' => $invoice_date,
                    'remarks' => $remarks,
                    'uuid' => $eachRowUUID,
                    'processedBy' => auth()->id(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                // voucher preparation
                $subAutoAcNo = '';
                $subDivName = '';
                $subDivCode = '';

                $fndCustomer = $bill->customer;
                $custPKID = $fndCustomer->id;
                $customer_name = $fndCustomer->customer_name;
                $custAutoAC = $fndCustomer->autoAcNo;
                if($bill->billSubject){
                    $subAutoAcNo = $bill->billSubject->autoAcNo;
                    $subDivName = $bill->billSubject->divName;
                    $subDivCode = $bill->billSubject->divCode;
                }

                $vatAutoAcNo = '';
                $vatDivCode = '';
                $vatDivName = '';
                $vatAmount = $bill->conv_vat_amount;
                if(round($vatAmount)>0){
                    $vatAutoAcNo = $vatAutoAcNoAPI;
                    $vatDivCode = $vatDivCodeAPI;
                    $vatDivName = $vatDivNameAPI;
                }
                $cusDivCode = $bill->department_id;
                $cusDivName = $bill->billDivName;

                // if($bill->ref_date){
                //     $formatedDate = DateTime::createFromFormat('d/m/Y', $invoice_date)->format('Y-m-d');
                // }else{
                //     $formatedDate = date('Y-m-d');
                // }

                // $formatedDate = DateTime::createFromFormat('d/m/Y', $invoice_date)->format('Y-m-d');
                $formatedDate = Carbon::createFromFormat('d/m/Y', $invoice_date)->format('Y-m-d');

                ($vatAutoAcNo)? $VAAT_NA = '||Bill for '.$customer_name : $VAAT_NA = '';
                ($vatAutoAcNo)? $VAAT_A = ','.$vatAutoAcNo : $VAAT_A = '';
                ($vatAutoAcNo)? $VAAT_D = ','.$vatAmount : $VAAT_D = '';
                ($vatAutoAcNo)? $VAAT_C = ',0' : $VAAT_C = '';
                // ($vatAutoAcNo)? $VAAT_N = '||'.$vatDivName : $VAAT_N = '';
                // ($vatAutoAcNo)? $VAAT_CO = ','.$vatDivCode : $VAAT_CO = '';
                ($vatAutoAcNo)? $VAAT_N = '||'.$cusDivName : $VAAT_N = '';
                ($vatAutoAcNo)? $VAAT_CO = ','.$cusDivCode : $VAAT_CO = '';

                $billProcessId = $eachRowUUID;

                if($voucherSettings->value=='yes'){

                    try{
                        $response = Http::withHeaders([
                            'Authorization' => 'Bearer ' . $token,
                        ])->post($api_ip.'/api/v1/voucherEntry/saveVoucherInfo', [
                            'voucherDt' => $formatedDate,
                            'voucherFlag' => 'JV',
                            'narration' =>  "Bill for $customer_name$VAAT_NA||Bill for $customer_name",
                            'verifyFlag' => 'N',
                            'autoAcNo' => $subAutoAcNo.$VAAT_A.','.$custAutoAC,
                            'debitAmt' => "$bill->convertion_total_amount".$VAAT_D.",0",
                            'creditAmt' => "0".$VAAT_C.",".$bill->convertion_total_amount_vat,
                            'companyNo' => $company_number,
                            'divName' => "$subDivName$VAAT_N||$cusDivName",
                            'divCode' => "$subDivCode$VAAT_CO,$cusDivCode",
                            'erpTranType' => 'billProcess',
                            'erpTranId' => $billProcessId,
                        ]);

                        if (!$response->successful()) {
                            // $errorMessage .= "Status: " . $response->status() . ". ";
                            // $errorMessage .= "Body: " . $response->body();
                            // return $errorMessage;
                            // if($response->status()==500){ $voucherError = true; }
                        }

                    }catch (\Exception $e) {
                        // if($e->getMessage()){ $voucherError = true; }
                    }
                }

            }
            $serial++;
        }

        try {
            DB::beginTransaction();
            // BillProcess::insert($billsArray);

            $chunkSize = 350;
            $billsChunks = array_chunk($billsArray, $chunkSize);

            foreach ($billsChunks as $chunk) {
                BillProcess::insert($chunk);
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            if($e->getMessage()){
                // return $processSLFormat.$processUUID;
                var_dump($e->getMessage());
                exit;
                return redirect()->route('bill-process.index')->with('error', 'Bill not processed.');
            }
        }

        $endTime = microtime(true);
        // return $executionTime = $endTime - $startTime;

        return redirect()->route('invoices.index')
            ->with('success', 'Bill processed successfull. Process No:   '.$processSLFormat);
    }

    public function generateInvoiceNumber($lastInvNo,$lastRefund,$entryDate,$entryRefund) {
        if($lastInvNo==''){
            if($entryRefund=='1'){
                return date('ym').'00000001';
            }else{
                return '00000001'.date('my');
            }
        }
        $currentYear = date('y',strtotime($entryDate));
        $currentMonth = date('m',strtotime($entryDate));

        // Parse the last invoice number 240800000001, 000000010824
        if($lastRefund=='1'){
            $year = substr($lastInvNo, 0, 2);
            $month = substr($lastInvNo, 2, 2);
            $serial = (int) substr($lastInvNo, 4);
        }else{
            $year = substr($lastInvNo, 10, 2);
            $month = substr($lastInvNo, 8, 2);
            $serial = (int) substr($lastInvNo, 0,8);
        }

        // If the year has changed or it's July or later in the current year, reset the serial
        if ($year != $currentYear || ($month < 7 && $currentMonth >= 7)) {
            $serial = 1;
        } else {
            $serial++;
        }
        if($entryRefund=='1'){
            $newInvNo = sprintf('%02d%02d%08d', $currentYear, $currentMonth, $serial);
        }else{
            $newInvNo = sprintf('%08d%02d%02d', $serial, $currentMonth, $currentYear);
        }

        return $newInvNo;
    }


    private function normalizeDate($date)
    {
        $formats = [
            'd/m/y',
            'd/m/Y',
            'd-M-Y',
            'd-M-y'
        ];

        foreach ($formats as $format) {
            try {
                $parsedDate = \Carbon\Carbon::createFromFormat($format, $date);
                return $parsedDate->format('d/m/Y');
            } catch (\Exception $e) {
            }
        }

        return $date;
    }

    public function sendProcessedBill()
    {
        try {
            // Mail::to('shakil.techeureka@gmail.com')->send(new ProcessedBill());

            return 'Static email sent successfully';
        } catch (\Exception $e) {
            return 'Failed to send static email: ' . $e->getMessage();
        }
    }

    public function singleBillProcess(Request $request){
        $id = $request->billID;
        $bill = Bill::with([
            'billSubject:id,divCode,divName,autoAcNo,bill_subject',
            'customer:id,customer_name,autoAcNo',
        ])->find($id);
        $remarks = trim($request->remarks);

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        if($token==''){
            return 0;
        }

        $processSLNo = BillProcess::max('process_no')+1;
        $processSLFormat = sprintf('%08d%03d', date('Ymd'), $processSLNo);
        $processUUID = mt_rand(111111, 999999);
        $eachRowUUID = mt_rand(11111111, 99999999);

        $invoice_date = $request->invoice_date;
        $entryDate = date('Y-m-d');
        $entryRefund = $bill->vat_refund;

        // Invoice Serial Generate
        $lastRecord = BillProcess::select(['invoice_no', 'vat_refund'])
            ->where('invoice_no', '!=', '')
            ->whereNotNull('invoice_no')
            ->orderByDesc('id')
            ->first();

        $lastInvNo = $lastRecord->invoice_no ?? '';
        $lastRefund = $lastRecord->vat_refund ?? '';
        $invoice_no = $this->generateInvoiceNumber($lastInvNo,$lastRefund,$entryDate,$entryRefund);

        $data = BillProcess::create([
            'bill_id' => $bill->id,
            'customer_id' => $bill->customer_id,
            'bill_subject_id' => $bill->bill_subject_id,
            'particulars' => $bill->particulars,
            'bill_subject_desc' => ($bill->billSubject) ? $bill->billSubject->bill_subject . ' ' . $remarks : null,
            'currency' => $bill->currency,
            'currency_rate' => $bill->currency_rate,
            'qty' => $bill->qty,
            'unit_price_vat' => $bill->unit_price_vat,
            'unit_total' => $bill->unit_total,
            'conv_unit_total' => $bill->conv_unit_total,
            'total_amount' => $bill->total_amount,
            'convertion_total_amount' => $bill->convertion_total_amount,
            'vat_type' => $bill->vat_type,
            'vat_status' => $bill->vat_status,
            'vat_percent' => $bill->vat_percent,
            'vat_amount' => $bill->vat_amount,
            'conv_vat_amount' => $bill->conv_vat_amount,
            'total_amount_vat' => $bill->total_amount_vat,
            'convertion_total_amount_vat' => $bill->convertion_total_amount_vat,
            'due' => $bill->total_amount_vat,

            'ref_no' => $bill->ref_no,
            'ref_date' => $bill->ref_date,
            'installation_date' => $bill->installation_date,
            'activation_date' => $bill->activation_date,
            'start_date' => $bill->start_date,
            'end_date' => $bill->end_date,
            'attention_id' => $bill->attention_id,
            'attention_person' => $bill->attention_person,
            'department_id' => $bill->department_id,
            'billDivName' => $bill->billDivName,
            'bill_group' => $bill->bill_group,
            'bill_type' => $bill->bill_type,
            'payment_type' => $bill->payment_type,
            'vat_refund' => ($entryRefund)? '1':'0',
            'yearly_increase_percent' => $bill->yearly_increase_percent,
            'increase_start_date' => $bill->increase_start_date,
            'payment_status' => 'Unpaid',
            'hs_code' => $bill->hs_code,
            'order_sl_no' => $bill->order_sl_no,
            'month' => $request->month,
            'year' => $request->year,
            'invoice_no' => $invoice_no,
            'process_no' => $processSLNo,
            'process_no_format' => $processSLFormat,
            'process_uuid' => $processUUID,
            'invoice_date' => $invoice_date,
            'remarks' => $remarks,
            'uuid' => $eachRowUUID,
            'processedBy' => auth()->id(),
        ]);

        $subAutoAcNo = '';
        $subDivName = '';
        $subDivCode = '';

        if($bill->billSubject){
            $subAutoAcNo = $bill->billSubject->autoAcNo;
            $subDivName = $bill->billSubject->divName;
            $subDivCode = $bill->billSubject->divCode;
        }

        $fndCustomer = $bill->customer;
        $custPKID = $fndCustomer->id;
        $customer_name = $fndCustomer->customer_name;
        $custAutoAC = $fndCustomer->autoAcNo;

        $vatAutoAcNo = '';
        $vatDivCode = '';
        $vatDivName = '';

        $vatAmount = $bill->conv_vat_amount;

        if(round($vatAmount)>0){

            // TDS/TAX Amount
            $getVatHead = GlHead::where('gl_subject',strtoupper('VAT'))->first();
            if($getVatHead){
                $glCodeVat = $getVatHead->gl_code;
                try{
                    $response4 = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                        'groupCode' => $glCodeVat,
                        'companyNo' => $company_number,
                    ]);

                    if ($response4->successful()) {
                        $responseData4 = $response4->json();
                        if ($responseData4 && isset($responseData4['accHeadDetailsInfo'])) {

                            $vatAutoAcNo = $responseData4['accHeadDetailsInfo']['autoAcNo'];
                            $vatDivCode = $responseData4['accHeadDetailsInfo']['divCode'];
                            $vatDivName = $responseData4['accHeadDetailsInfo']['divName'];
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }

        $cusDivCode = $bill->department_id;
        $cusDivName = $bill->billDivName;

        // if($bill->ref_date){
        //     $formatedDate = DateTime::createFromFormat('d/m/Y', $invoice_date)->format('Y-m-d');
        // }else{
        //     $formatedDate = date('Y-m-d');
        // }
        $formatedDate = DateTime::createFromFormat('d/m/Y', $invoice_date)->format('Y-m-d');

        ($vatAutoAcNo)? $VAAT_NA = '||Bill for '.$customer_name : $VAAT_NA = '';
        ($vatAutoAcNo)? $VAAT_A = ','.$vatAutoAcNo : $VAAT_A = '';
        ($vatAutoAcNo)? $VAAT_D = ','.$vatAmount : $VAAT_D = '';
        ($vatAutoAcNo)? $VAAT_C = ',0' : $VAAT_C = '';
        // ($vatAutoAcNo)? $VAAT_N = '||'.$vatDivName : $VAAT_N = '';
        // ($vatAutoAcNo)? $VAAT_CO = ','.$vatDivCode : $VAAT_CO = '';
        ($vatAutoAcNo)? $VAAT_N = '||'.$cusDivName : $VAAT_N = '';
        ($vatAutoAcNo)? $VAAT_CO = ','.$cusDivCode : $VAAT_CO = '';

        $billProcessId = $eachRowUUID;
        try{
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->post($api_ip.'/api/v1/voucherEntry/saveVoucherInfo', [
                'voucherDt' => $formatedDate,
                'voucherFlag' => 'JV',
                'narration' =>  "Bill for $customer_name$VAAT_NA||Bill for $customer_name",
                'verifyFlag' => 'N',
                'autoAcNo' => $subAutoAcNo.$VAAT_A.','.$custAutoAC,
                'debitAmt' => "$bill->convertion_total_amount".$VAAT_D.",0",
                'creditAmt' => "0".$VAAT_C.",".$bill->convertion_total_amount_vat,
                'companyNo' => $company_number,
                'divName' => "$subDivName$VAAT_N||$cusDivName",
                'divCode' => "$subDivCode$VAAT_CO,$cusDivCode",
                'erpTranType' => 'billProcess',
                'erpTranId' => $billProcessId,
            ]);

            if (!$response->successful()) {
                $data->delete();
                echo 0;
            }else{

                // Customer Ledger
                // $ledger                 = new CustomerLedger();
                // $ledger->entry_date     = date('Y-m-d');
                // $ledger->narration      = 'Invoice';
                // $ledger->customer_id    = $bill->customer_id;
                // $ledger->type           = 'Invoice';
                // $ledger->ref_id         = null;
                // $ledger->ref_no         = $invoice_no;
                // $ledger->amount         = $bill->total_amount_vat;
                // $ledger->created_by     = auth()->id();
                // $ledger->save();

                echo 1;

            }

        }catch (\Exception $e) {
            // Log::error($e->getMessage());
            $data->delete();
            echo 0;
        }

        // $data->load('customer', 'billSubject');

        // // return view('emails.processed_bill', ['billProcess' => $data,'settings'=> $settings]);

        // try {
        //     if($bill->customer->billing_email){
        //         Mail::to($bill->customer->billing_email)->send(new ProcessedBill($data, $settings));
        //     }

        // } catch (\Exception $e) {
        //     Log::error('Failed to send email: ' . $e->getMessage());
        // }

        // $successURL = $request->successURL;
        // return redirect($successURL)->with('success', 'BIll processed successfully.');

    }


    // Cancel Invoice
    public function cancelInvoicePreview(Request $request)
    {

        $query = BillProcess::query();

        if ($request->filled('invoice_no')) {
            $query->where('invoice_no', $request->invoice_no);
            $query->where('delete_flag', 0);
            $data = $query->get();
        }else{
            $data = [];
            return view('invoice.cancel-invoice-preview', [
                'invoices' => [],
                'firstCustomerName' => '',
                'firstCustomerBinNo' => '',
                'firstBillSubject' => [],
                'invoiceNo' => '',
                'invoiceDate' => '',
                'address' => '',
                'currency' => '',
                'billData' => $data,
                'collectionList' => ''
            ]);
        }


        $collectionArr = [];
        $billID = [];
        foreach ($data as $value) {
            $billID[] = $value->id;

            // collection linked with this invoice
            $collectionInfo = CollectionChd::where('bill_process_id',$value->id)
                ->where('delete_flag',0)->get();
            if($collectionInfo){
                foreach($collectionInfo as $row){
                    $collectionArr[] = $row->ref_no;
                }
            }
        }
        $collectionList = implode(', ', array_unique($collectionArr));

        $billIds = implode(',', $billID);

        // bellow are old defined function for grouped invoice (showGrouped())

        $billIdsArray = explode(',', $billIds);
        $invoices = BillProcess::whereIn('id', $billIdsArray)->get();

        $customerIds = $invoices->pluck('customer_id')->unique();
        $customers = \App\Models\Customer::whereIn('id', $customerIds)->get()->keyBy('id');

        $customerNames = $invoices->map(function ($invoice) {
            return $invoice->customer ? $invoice->customer->customer_name : '';
        })->unique();

        $billSubjectIds = $invoices->pluck('bill_subject_id')->unique();
        $firstBillSubject = BillSubject::find($billSubjectIds->first());

        $firstCustomerName = $customerNames->first();

        // Retrieve the first invoice's number and date for display
        $firstInvoice = $invoices->first();
        $invoiceNo = $firstInvoice ? $firstInvoice->invoice_no : '';
        $invoiceDate = $firstInvoice ? $firstInvoice->invoice_date : '';
        $currency = $firstInvoice ? $firstInvoice->currency : '';

        // Retrieve the address for the first customer
        $firstCustomerId = $invoices->first()->customer_id ?? '';
        $address = $firstCustomerId ? ($customers->get($firstCustomerId)->address ?? '') : '';
        $firstCustomerBinNo = $firstCustomerId ? ($customers->get($firstCustomerId)->bin_no ?? '') : '';

        return view('invoice.cancel-invoice-preview', [
            'invoices' => $invoices,
            'firstCustomerName' => $firstCustomerName,
            'firstCustomerBinNo' => $firstCustomerBinNo,
            'firstBillSubject' => $firstBillSubject,
            'invoiceNo' => $invoiceNo,
            'invoiceDate' => $invoiceDate,
            'address' => $address,
            'currency' => $currency,
            'billData' => $data,
            'collectionList' => $collectionList
        ]);

    }

    public function cancelInvoiceDetails($invoiceNo)
    {
        // BillProcess::where('invoice_no',$invoiceNo)->update(['delete_flag'=>'1']);

        $insertLedger = false;

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $customerID = null;
        $amount = 0;
        $bills = BillProcess::where('invoice_no',$invoiceNo)->get();

        foreach ($bills as $bill) {
            $bill->bill_id = null;
            $bill->invoice_no = null;
            $bill->process_no = null;
            $bill->process_no_format = null;
            $bill->sl_no = null;
            $bill->delete_flag = 1;

            $amount = $amount+$bill->total_amount_vat;
            $erpTranId = $bill->uuid;

            try{

                $response4 = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/voucherEntry/voucherInfo/billProcess/'.$erpTranId.'/'.$company_number);

                if ($response4->successful()) {

                    $responseData4 = $response4->json();
                    if(!$responseData4){

                        $bill->save();
                        return redirect()->route('cancel.invoice.preview')->with('success', 'Invoice Cancelled successfully.');

                        // return redirect()->route('cancel.invoice.preview')->with('error', 'Invoice Cancellation not possible.');
                    }
                    if ($responseData4 && isset($responseData4['voucherMstId'])) {
                        $voucherMstId = $responseData4['voucherMstId'];
                        if($voucherMstId==''){
                            return redirect()->route('cancel.invoice.preview')->with('error', 'Invoice Cancellation not possible.');
                        }
                    }

                    if($voucherMstId){
                        // delete voucher procedure from api
                        $response = Http::withHeaders([
                            'Authorization' => 'Bearer ' . $token,
                        ])->delete($api_ip.'/api/v1/deleteVoucher/'.$voucherMstId);

                        if($response->successful()){
                            $bill->save();
                            $insertLedger = true;
                        }
                    }

                }
            }catch (\Exception $e) {
                return redirect()->route('cancel.invoice.preview')->with('error', 'Invoice Cancellation not possible.');
            }

        }

        if($insertLedger){
            // Customer Ledger
            // $ledger                 = new CustomerLedger();
            // $ledger->entry_date     = date('Y-m-d');
            // $ledger->narration      = 'Invoice Cancel';
            // $ledger->customer_id    = $bill->customer_id;
            // $ledger->type           = 'Invoice-cancel';
            // $ledger->ref_id         = null;
            // $ledger->ref_no         = $invoiceNo;
            // $ledger->amount         = -$amount;
            // $ledger->created_by     = auth()->id();
            // $ledger->save();

            return redirect()->route('cancel.invoice.preview')->with('success', 'Invoice Cancelled successfully.');
        }else{
            return redirect()->route('cancel.invoice.preview')->with('error', 'Invoice cancellation not possible.');
        }

    }

    // Old Bill Process - OLD STYLE WORKING FROM START DATE
    public function billProcessOldForm(Request $request)
    {
        $customers = Customer::select('id','customer_name','autoAcNo')
            ->where('status','Active')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('bill-process.billProcessOldForm',compact('customers'));
    }


    // this is perfectly working - OLD STYLE WORKING FROM START DATE
    public function findBillsOld(Request $request)
    {
        if (!auth()->user()->can('Process Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $startTime = microtime(true);

        $billfilters=$request;
        $customer_id=$request->customer;
        $customerInfo = Customer::select('customer_name')->find($customer_id);

        if($request->bill_type=='One Time'){
            $requestedMonth = "{$request->month}/{$request->year}";

            $startOfMonth = Carbon::createFromFormat('F/Y', $requestedMonth)->startOfMonth()->format('Y-m-d');
            $endOfMonth = Carbon::createFromFormat('F/Y', $requestedMonth)->endOfMonth()->format('Y-m-d');

            $data = Bill::select([
                'id',
                'start_date',
                'end_date',
                'bill_subject_id',
                'bill_group',
                'qty',
                'unit_price_vat',
                'currency',
                'currency_rate',
                'total_amount',
                'vat_amount',
                'total_amount_vat',
                'bill_type'
            ])->where('bill_type', 'One Time')
            ->where('status', 'Active')
            ->where('approval', '1')
            ->where('customer_id', $customer_id)
            ->where('vat_refund', $request->vat_refund)
            ->where(function($query) use ($startOfMonth, $endOfMonth) {
                $query->whereRaw("
                    STR_TO_DATE(start_date, '%d/%m/%Y') BETWEEN ? AND ?
                    OR STR_TO_DATE(end_date, '%d/%m/%Y') BETWEEN ? AND ?
                ", [$startOfMonth, $endOfMonth, $startOfMonth, $endOfMonth]);
            })->get();

            $bills=[];

            foreach ($data as $value) {
                $processedBill=BillProcess::where('bill_id', $value->id)->get();

                if($processedBill->isEmpty()){
                    $billCopy = clone $value;
                    $billCopy['monthNo'] = $request->month;
                    $billCopy['yearNo'] = $request->year;
                    $bills[] = $billCopy;
                }
            }

        }else{
            // Recurring bills
            $bills=[];
            $targetDate = \DateTime::createFromFormat('F Y', $request->month . ' ' . $request->year);

            $dataMonthly = Bill::select([
                    'id',
                    'start_date',
                    'end_date',
                    'bill_subject_id',
                    'bill_group',
                    'qty',
                    'unit_price_vat',
                    'currency',
                    'currency_rate',
                    'total_amount',
                    'vat_amount',
                    'total_amount_vat',
                    'bill_type'
                ])->where('bill_type', 'Monthly')
                ->where('status', 'Active')
                ->where('approval', '1')
                ->where('customer_id', $customer_id)
                ->where('vat_refund', $request->vat_refund)
                ->where(function ($query) use ($request) {
                    $startOfMonth = date("Y-m-d", strtotime("last day of {$request->year}-{$request->month}"));

                    $query->where(DB::raw("STR_TO_DATE(start_date, '%d/%m/%Y')"), '<=', $startOfMonth);
                    })->get();

            foreach ($dataMonthly as $value) {

                $start_date = \DateTime::createFromFormat('d/m/Y', $value->start_date);
                $end_date = \DateTime::createFromFormat('d/m/Y', $value->end_date);

                $periods = [];

                $current_date = clone $start_date;
                while ($current_date <= $end_date) {
                    $periods[] = [
                        'month' => $current_date->format('F'),
                        'year' => $current_date->format('Y')
                    ];

                    $current_date->modify('+1 months');
                }

                $result = array_filter($periods, function ($item) use ($targetDate) {
                    $itemDate = \DateTime::createFromFormat('F Y', $item['month'] . ' ' . $item['year']);
                    return $itemDate <= $targetDate; // Compare the dates
                });
                $periods = $result;

                foreach ($periods as $period) {
                    $processData = BillProcess::where('bill_id',$value->id)
                        ->where('month',$period['month'])
                        ->where('year',$period['year'])
                        ->first();
                    if(!$processData){
                        $billCopy = clone $value;
                        $billCopy['monthNo'] = $period['month'];
                        $billCopy['yearNo'] = $period['year'];
                        $bills[] = $billCopy;
                    }
                }
            }


            $dataQuarterly = Bill::select([
                    'id',
                    'start_date',
                    'end_date',
                    'bill_subject_id',
                    'bill_group',
                    'qty',
                    'unit_price_vat',
                    'currency',
                    'currency_rate',
                    'total_amount',
                    'vat_amount',
                    'total_amount_vat',
                    'bill_type'
                ])->where('bill_type', 'Quarterly')
                ->where('status', 'Active')
                ->where('approval', '1')
                ->where('customer_id', $customer_id)
                ->where('vat_refund', $request->vat_refund)
                ->where(function ($query) use ($request) {
                    $startOfMonth = date("Y-m-d", strtotime("last day of {$request->year}-{$request->month}"));

                    $query->where(DB::raw("STR_TO_DATE(start_date, '%d/%m/%Y')"), '<=', $startOfMonth);
                    })->get();


            foreach ($dataQuarterly as $value) {
                $start_date = \DateTime::createFromFormat('d/m/Y', $value->start_date);
                $end_date = \DateTime::createFromFormat('d/m/Y', $value->end_date);

                $periods = [];

                $current_date = clone $start_date;
                while ($current_date <= $end_date) {
                    $periods[] = [
                        'month' => $current_date->format('F'),
                        'year' => $current_date->format('Y')
                    ];

                    $current_date->modify('+3 months');
                }

                $result = array_filter($periods, function ($item) use ($targetDate) {
                    $itemDate = \DateTime::createFromFormat('F Y', $item['month'] . ' ' . $item['year']);
                    return $itemDate <= $targetDate; // Compare the dates
                });
                $periods = $result;

                foreach ($periods as $period) {
                    $processData = BillProcess::where('bill_id',$value->id)
                        ->where('month',$period['month'])
                        ->where('year',$period['year'])
                        ->first();
                    if(!$processData){
                        $billCopy = clone $value;
                        $billCopy['monthNo'] = $period['month'];
                        $billCopy['yearNo'] = $period['year'];
                        $bills[] = $billCopy;
                    }
                }
            }

            $dataHYearly = Bill::select([
                    'id',
                    'start_date',
                    'end_date',
                    'bill_subject_id',
                    'bill_group',
                    'qty',
                    'unit_price_vat',
                    'currency',
                    'currency_rate',
                    'total_amount',
                    'vat_amount',
                    'total_amount_vat',
                    'bill_type'
                ])->where('bill_type', 'Half Yearly')
                ->where('status', 'Active')
                ->where('approval', '1')
                ->where('customer_id', $customer_id)
                ->where('vat_refund', $request->vat_refund)
                ->where(function ($query) use ($request) {
                    $startOfMonth = date("Y-m-d", strtotime("last day of {$request->year}-{$request->month}"));

                    $query->where(DB::raw("STR_TO_DATE(start_date, '%d/%m/%Y')"), '<=', $startOfMonth);
                    })->get();

            foreach ($dataHYearly as $value) {
                $start_date = \DateTime::createFromFormat('d/m/Y', $value->start_date);
                $end_date = \DateTime::createFromFormat('d/m/Y', $value->end_date);

                $periods = [];

                $current_date = clone $start_date;
                while ($current_date <= $end_date) {
                    $periods[] = [
                        'month' => $current_date->format('F'),
                        'year' => $current_date->format('Y')
                    ];

                    $current_date->modify('+6 months');
                }

                $result = array_filter($periods, function ($item) use ($targetDate) {
                    $itemDate = \DateTime::createFromFormat('F Y', $item['month'] . ' ' . $item['year']);
                    return $itemDate <= $targetDate; // Compare the dates
                });
                $periods = $result;

                foreach ($periods as $period) {
                    $processData = BillProcess::where('bill_id',$value->id)
                        ->where('month',$period['month'])
                        ->where('year',$period['year'])
                        ->first();
                    if(!$processData){
                        $billCopy = clone $value;
                        $billCopy['monthNo'] = $period['month'];
                        $billCopy['yearNo'] = $period['year'];
                        $bills[] = $billCopy;
                    }
                }
            }

            $dataYearly = Bill::select([
                    'id',
                    'start_date',
                    'end_date',
                    'bill_subject_id',
                    'bill_group',
                    'qty',
                    'unit_price_vat',
                    'currency',
                    'currency_rate',
                    'total_amount',
                    'vat_amount',
                    'total_amount_vat',
                    'bill_type'
                ])->where('bill_type', 'Yearly')
                ->where('status', 'Active')
                ->where('approval', '1')
                ->where('customer_id', $customer_id)
                ->where('vat_refund', $request->vat_refund)
                ->where(function ($query) use ($request) {
                    $startOfMonth = date("Y-m-d", strtotime("last day of {$request->year}-{$request->month}"));

                    $query->where(DB::raw("STR_TO_DATE(start_date, '%d/%m/%Y')"), '<=', $startOfMonth);
                    })->get();

            foreach ($dataYearly as $value) {
                $start_date = \DateTime::createFromFormat('d/m/Y', $value->start_date);
                $end_date = \DateTime::createFromFormat('d/m/Y', $value->end_date);

                $periods = [];

                $current_date = clone $start_date;
                while ($current_date <= $end_date) {
                    $periods[] = [
                        'month' => $current_date->format('F'),
                        'year' => $current_date->format('Y')
                    ];

                    $current_date->modify('+12 months');
                }

                $result = array_filter($periods, function ($item) use ($targetDate) {
                    $itemDate = \DateTime::createFromFormat('F Y', $item['month'] . ' ' . $item['year']);
                    return $itemDate <= $targetDate; // Compare the dates
                });
                $periods = $result;

                foreach ($periods as $period) {
                    $processData = BillProcess::where('bill_id',$value->id)
                        ->where('month',$period['month'])
                        ->where('year',$period['year'])
                        ->first();
                    if(!$processData){
                        $billCopy = clone $value;
                        $billCopy['monthNo'] = $period['month'];
                        $billCopy['yearNo'] = $period['year'];
                        $bills[] = $billCopy;
                    }
                }
            }

        }

        $endTime = microtime(true);
        $executionTime = $endTime - $startTime;

        return view('bill-process.billProcessOld', compact('bills','billfilters','executionTime','customerInfo'));
    }

    // Old Bill Process
    public function billProcessUnpostedForm(Request $request)
    {
        $customers = Customer::select('id','customer_name','autoAcNo')
            ->where('status','Active')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('bill-process.billProcessUnpostedForm',compact('customers'));
    }

    // This is working
    public function findBillsUnposted(Request $request)
    {
        if (!auth()->user()->can('Process Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $startTime = microtime(true);

        $billfilters = $request;
        $customer_id=$request->customer;
        $customerInfo = Customer::select('customer_name')->find($customer_id);

        $bills = [];
        if ($request->bill_type == 'One Time') {
            $requestedMonth = "{$request->month}/{$request->year}";
            $startOfMonth = Carbon::createFromFormat('F/Y', $requestedMonth)->startOfMonth()->format('Y-m-d');
            $endOfMonth = Carbon::createFromFormat('F/Y', $requestedMonth)->endOfMonth()->format('Y-m-d');
            $data = Bill::select('id', 'bill_type', 'status', 'approval', 'vat_refund', 'start_date', 'end_date',
                'customer_id','bill_group','qty','unit_price_vat','currency','currency_rate','convertion_total_amount',
                'conv_vat_amount','convertion_total_amount_vat','bill_type','bill_subject_id','billDivName')
                ->where('bill_type', 'One Time')
                ->where('status', 'Active')
                ->where('approval', '1')
                ->where('customer_id', $customer_id)
                ->where('vat_refund', $request->vat_refund)
                ->where(function ($query) use ($startOfMonth, $endOfMonth) {
                    $query->whereRaw(
                        "STR_TO_DATE(start_date, '%d/%m/%Y') BETWEEN ? AND ?
                        OR STR_TO_DATE(end_date, '%d/%m/%Y') BETWEEN ? AND ?",
                        [$startOfMonth, $endOfMonth, $startOfMonth, $endOfMonth]
                    );
                })
                ->with(['billProcesses:id,bill_id,month,year'])
                ->with(['billSubject:id,bill_subject'])
                ->with(['customer:id,customer_name,autoAcNo'])
                ->get();
            foreach ($data as $value) {
                if ($value->billProcesses->isEmpty()) {
                    $bills[] = $value;
                }
            }
        } else {
            $recurringTypes = ['Monthly', 'Quarterly', 'Half Yearly', 'Yearly'];
            foreach ($recurringTypes as $type) {
                $interval = match ($type) {
                    'Monthly' => 1,
                    'Quarterly' => 3,
                    'Half Yearly' => 6,
                    'Yearly' => 12,
                    default => 1,
                };
                $data = Bill::select('id', 'bill_type', 'status', 'approval', 'vat_refund', 'start_date', 'end_date',
                    'customer_id','bill_group','qty','unit_price_vat','currency','currency_rate','convertion_total_amount',
                    'conv_vat_amount','convertion_total_amount_vat','bill_type','bill_subject_id','billDivName',
                    'total_amount','vat_amount','total_amount_vat')
                    ->where('bill_type', $type)
                    ->where('status', 'Active')
                    ->where('approval', '1')
                    ->where('customer_id', $customer_id)
                    ->where('vat_refund', $request->vat_refund)
                    ->where(function ($query) use ($request) {
                        $startOfMonth = date("Y-m-d", strtotime("last day of {$request->year}-{$request->month}"));
                        $query->where(DB::raw("STR_TO_DATE(start_date, '%d/%m/%Y')"), '<=', $startOfMonth);
                    })
                    ->with(['billProcesses:id,bill_id,month,year'])
                    ->with(['billSubject:id,bill_subject'])
                    ->with(['customer:id,customer_name,autoAcNo'])
                    ->get();
                foreach ($data as $value) {
                    $start_date = \DateTime::createFromFormat('d/m/Y', $value->start_date);
                    $end_date = \DateTime::createFromFormat('d/m/Y', $value->end_date);
                    if (!$start_date || !$end_date) {
                        continue;
                    }
                    $current_date = clone $start_date;
                    $billMatched = false;
                    while ($current_date <= $end_date) {
                        if ($current_date->format('F') === $request->month && $current_date->format('Y') == $request->year) {
                            $billMatched = true;
                            break;
                        }
                        $current_date->modify("+{$interval} months");
                    }
                    if ($billMatched) {
                        $processedBill = $value->billProcesses
                            ->where('month', $request->month)
                            ->where('year', $request->year);
                        if ($processedBill->isEmpty()) {
                            $bills[] = $value;
                        }
                    }
                }
            }
        }

        $executionTime = microtime(true) - $startTime;

        return view('bill-process.billProcessUnposted', compact('bills', 'billfilters','customerInfo','executionTime'));
    }


    // Unposted bill list Pdf generate
    public function billProcessPdfUnposted(Request $request){

        $i = 0;
        $data['req_month'] = $req_month = $request->req_month;
        $data['req_year'] = $req_year = $request->req_year;

        $custInfo = Customer::select('customer_name')->find($request->req_customer);
        $data['customerName'] = $customer_name = $custInfo->customer_name ?? '';

        $processStatus = $request->processStatus;
        $billID = $request->billID;
        $monthTitle = $request->monthTitle;

        $result = [];
        foreach ($processStatus as $statusVal) {
            if($statusVal!=1){

                $bill = Bill::select([
                        'id',
                        'customer_id',
                        'bill_subject_id',
                        'bill_group',
                        'qty',
                        'unit_price_vat',
                        'currency',
                        'currency_rate',
                        'total_amount',
                        'vat_amount',
                        'total_amount_vat',
                        'bill_type'
                    ])->with([
                        'billSubject:id,bill_subject',
                        // 'customer:id,customer_name',
                    ])->find($billID[$i]);

                $result[] = [
                    "bill_no" => str_pad($bill->id, 5, '0', STR_PAD_LEFT),
                    // "customer_name" => ($bill->customer)? $bill->customer->customer_name ?? '':'',
                    "customer_name" => $customer_name,
                    "bill_subject" => ($bill->billSubject)? $bill->billSubject->bill_subject ?? '':'',
                    "bill_group" => $bill->bill_group,
                    "qty" => $bill->qty,
                    "month_year" => $monthTitle[$i],
                    "unit_price" => priceFormat($bill->unit_price_vat),
                    "currency" => $bill->currency,
                    "currency_rate" => priceFormat($bill->currency_rate),
                    "total_amount" => priceFormat($bill->total_amount),
                    "vat_amount" => priceFormat($bill->vat_amount),
                    "total_amount_vat" => priceFormat($bill->total_amount_vat),
                    "bill_type" => $bill->bill_type
                ];
            }
            $i++;

        }
        $data['result'] = $result;
        $name = "Unposted_bill_report_".$req_month."_".$req_year.".pdf";
        $pdf = Pdf::loadView('bill-process.unposted-bill-pdf', $data);
        $pdf->setPaper('A4', 'landscape');
        // return $pdf->stream($name, ['Attachment'=>false]);
        return $pdf->download($name); // Force download

    }

    // VAT Challan List
    public function vatChallanList(Request $request)
    {
        if (!auth()->user()->can('Process Bill')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 50);

        $query = BillProcess::query();
        $query->where('challan_ref_no', '>', 0);

        // Apply filters
        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(challan_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(challan_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('challan_ref_no')) {
            $query->where('challan_ref_no', $request->challan_ref_no);
        }

        $groupedChallans = $query->with('generatedBy')
        ->select(
            'challan_ref_no',
            'challan_date',
            'challan_gen_by',
            'challan_remarks'
        )
        ->groupBy('challan_ref_no', 'challan_date', 'challan_gen_by', 'challan_remarks')
        ->orderBy(DB::raw('MAX(challan_ref_no)'), 'desc')
        ->paginate($perPage)
        ->appends($request->except('page'));

        return view('bill-process.vat-challan-list', compact('groupedChallans', 'perPage'));

    }

    // VAT Challan Generate Form
    public function vatChallanGenerate()
    {

        // $billProcesses = BillProcess::where('id', '>', 147233)
        $billProcesses = BillProcess::where('bill_old', '!=', 1)
            ->orWhereNull('bill_old')
            ->where('vat_refund', '1')
            ->where('challan_sl', 0)
            ->where('delete_flag', 0)
            ->select(
                'customer_id',
                'invoice_no',
                'invoice_date',
                'bill_subject_desc',
                DB::raw('SUM(total_amount) as total_amount'),
                DB::raw('SUM(vat_amount) as vat_amount'),
                DB::raw('SUM(total_amount_vat) as total_amount_vat')
            )
            ->with(['billSubject', 'customer']) // Load related Customer data
            ->groupBy('customer_id', 'invoice_no', 'invoice_date', 'bill_subject_desc')
            ->orderBy('customer_id')
            ->get()
            ->groupBy('customer_id');

        $result = [];
        foreach ($billProcesses as $customerId => $bills) {
            $result[$customerId]['rows'] = $bills;
            $result[$customerId]['total'] = [
                'total_amount' => $bills->sum('total_amount'),
                'vat_amount' => $bills->sum('vat_amount'),
                'total_amount_vat' => $bills->sum('total_amount_vat'),
            ];
            $result[$customerId]['customer_name'] = $bills->first()->customer->customer_name ?? '';
        }

        $nextChallanRefNo = (BillProcess::max('challan_ref_no') ?? 0) + 1;
        $lastChallanInfo = BillProcess::select('challan_date', 'challan_no')->orderBy('challan_sl', 'desc')->first();
        $lastChallanNo = optional($lastChallanInfo)->challan_no ?? '';
        $lastChallanDate = optional($lastChallanInfo)->challan_date ?? date('01/m/Y');

        return view('bill-process.vat-challan-generate', compact('result','nextChallanRefNo','lastChallanNo','lastChallanDate'));

    }

    // VAT Challan Generate Store
    // public function vatChallanStore(Request $request){

    //     $generated_by = auth()->id();
    //     $challan_ref_no = $request->challan_ref_no;
    //     $challan_date = $request->challan_date;
    //     $challan_remarks = $request->challan_remarks;
    //     $check_invoice = $request->checkInvoice;
    //     $challan_no = $request->challan_no;

    //     $max_challan_sl = (BillProcess::max('challan_sl') ?? 0) + 1;

    //     try {
    //         DB::beginTransaction();
    //         if (count($check_invoice) > 0) {
    //             $i = 0;
    //             foreach ($check_invoice as $invoice_no) {

    //                 $data_param = [
    //                     'challan_ref_no'   => $challan_ref_no,
    //                     'challan_date'     => $challan_date,
    //                     'challan_no'       => $challan_no[$i],
    //                     'challan_sl'       => $max_challan_sl,
    //                     'challan_gen_by'   => $generated_by,
    //                     'challan_remarks'  => $challan_remarks
    //                 ];

    //                 BillProcess::where('invoice_no', $invoice_no)->update($data_param);
    //                 $max_challan_sl++;
    //                 $i++;
    //             }
    //         }

    //         DB::commit();
    //         return redirect()->route('bill-process.vat-challan-list')
    //                          ->with('success', 'VAT Challan No Generated Successfully. VAT Challan Ref No: '.$challan_ref_no);
    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         return redirect()->back()
    //                          ->with('error', 'VAT Challan No Not Generated.');
    //     }

    // }

    public function vatChallanStore(Request $request)
    {
        $generated_by    = auth()->id();
        $challan_ref_no  = $request->challan_ref_no;
        $challan_date    = $request->challan_date;
        $challan_remarks = $request->challan_remarks;
        $check_invoice   = $request->checkInvoice;
        $challan_no_list = $request->challan_no;

        $max_challan_sl = (BillProcess::max('challan_sl') ?? 0) + 1;

        if (empty($check_invoice) || empty($challan_no_list)) {
            return redirect()->back()->with('error', 'No invoices selected for VAT Challan.');
        }

        try {
            DB::beginTransaction();

            foreach ($check_invoice as $index => $invoice_no) {
                $data = [
                    'challan_ref_no'   => $challan_ref_no,
                    'challan_date'     => $challan_date,
                    'challan_no'       => $challan_no_list[$index] ?? null,
                    'challan_sl'       => $max_challan_sl++,
                    'challan_gen_by'   => $generated_by,
                    'challan_remarks'  => $challan_remarks,
                ];

                BillProcess::where('invoice_no', $invoice_no)->update($data);
            }

            DB::commit();

            return redirect()->route('bill-process.vat-challan-list')
                            ->with('success', 'VAT Challan No Generated Successfully. VAT Challan Ref No: '.$challan_ref_no);
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'VAT Challan No Not Generated.');
        }
    }

    // VAT Challan View
    public function vatChallanView($challanRefNo)
    {

        $challanInfo = BillProcess::select('challan_ref_no','challan_date')->where('challan_ref_no', $challanRefNo)->first();

        $billProcesses = BillProcess::where('challan_ref_no', $challanRefNo)
        ->select(
            'customer_id',
            'invoice_no',
            'invoice_date',
            'bill_subject_desc',
            'challan_no',
            'challan_date',
            DB::raw('SUM(total_amount) as total_amount'),
            DB::raw('SUM(vat_amount) as vat_amount'),
            DB::raw('SUM(total_amount_vat) as total_amount_vat')
        )
        ->with(['billSubject', 'customer']) // Load related Customer data
        ->groupBy('customer_id', 'invoice_no', 'invoice_date', 'bill_subject_desc', 'challan_no', 'challan_date')
        ->orderBy('customer_id')
        ->get()
        ->groupBy('customer_id');

        $result = [];
        foreach ($billProcesses as $customerId => $bills) {
            $result[$customerId]['rows'] = $bills;
            $result[$customerId]['total'] = [
                'total_amount' => $bills->sum('total_amount'),
                'vat_amount' => $bills->sum('vat_amount'),
                'total_amount_vat' => $bills->sum('total_amount_vat'),
            ];
            $result[$customerId]['customer_name'] = $bills->first()->customer->customer_name ?? '';
            $result[$customerId]['address'] = $bills->first()->customer->address ?? '';
            $result[$customerId]['bin_no'] = $bills->first()->customer->bin_no ?? '';
        }

        return view('bill-process.vat-challan-view', compact('result','challanInfo'));

    }


    // VAT Challan Pdf Generate
    public function vatChallanViewPdf($challanRefNo){

        // $data['processNo'] = $processNo = str_replace("_","/",$processNo);

        // $data['processInfo'] = $pInfo = BillProcess::select('month','year')->where('process_no_format', $processNo)->first();

        $data['challanInfo'] = $challanInfo = BillProcess::select('challan_ref_no','challan_date')->where('challan_ref_no', $challanRefNo)->first();

        $billProcesses = BillProcess::where('challan_ref_no', $challanRefNo)
        ->select(
            'customer_id',
            'invoice_no',
            'invoice_date',
            'bill_subject_desc',
            'challan_no',
            'challan_date',
            DB::raw('SUM(total_amount) as total_amount'),
            DB::raw('SUM(vat_amount) as vat_amount'),
            DB::raw('SUM(total_amount_vat) as total_amount_vat')
        )
        ->with(['billSubject', 'customer']) // Load related Customer data
        ->groupBy('customer_id', 'invoice_no', 'invoice_date', 'bill_subject_desc', 'challan_no', 'challan_date')
        ->orderBy('customer_id')
        ->get()
        ->groupBy('customer_id');

        $result = [];
        foreach ($billProcesses as $customerId => $bills) {
            $result[$customerId]['rows'] = $bills;
            $result[$customerId]['total'] = [
                'total_amount' => $bills->sum('total_amount'),
                'vat_amount' => $bills->sum('vat_amount'),
                'total_amount_vat' => $bills->sum('total_amount_vat'),
            ];
            $result[$customerId]['customer_name'] = $bills->first()->customer->customer_name ?? 'N/A';
            $result[$customerId]['address'] = $bills->first()->customer->address ?? '';
            $result[$customerId]['bin_no'] = $bills->first()->customer->bin_no ?? '';
        }

        $data['result'] = $result;
        $challanDate = str_replace("/","-",$challanInfo->challan_date);
        $name = "VAT_Challan_Report_for_".$challanDate.".pdf";
        $pdf = Pdf::loadView('bill-process.vat-challan-view-pdf', $data);
        $pdf->setPaper('A4', 'landscape');
        // return $pdf->stream($name, ['Attachment'=>false]);
        return $pdf->download($name); // Force download

    }




}

