<?php
namespace App\Http\Controllers;
use DB;
use Carbon\Carbon;
use App\Models\Bill;
use App\Models\GlHead;
use App\Models\Customer;
use App\Models\CustomerDebit;
use App\Models\Settings;
use App\Models\Collection;
use App\Models\CollectionChd;
use App\Models\BillProcess;
use App\Models\CustomerBank;
use Illuminate\Http\Request;
use App\Models\DepositoryBank;
use App\Models\AdvancedCollection;
use App\Models\AdvancedCustomer;
use App\Models\CustomerLedger;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;
use DateTime;

class CollectionController extends Controller
{
    public function __construct()
    {
        set_time_limit(0);
        ini_set('memory_limit', '2000M');
    }

    public function index()
    {
        if (!auth()->user()->can('View Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $collections = Collection::all();
        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();
        return view('collection.index', compact('collections','customers'));
    }

    public function collectionList(Request $request)
    {
        if (!auth()->user()->can('View Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 50);

        $query = Collection::query();
        $query->where('delete_flag', 0);

        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('customer')) {
            $query->where('customer_id', $request->customer);
        }

        if ($request->filled('ref_no')) {
            $query->where('ref_no', $request->ref_no);
        }

        if ($request->filled('instrument_mode')) {
            if($request->instrument_mode!='All'){
                if($request->instrument_mode=='Advance'){
                    $query->where('instrument_mode','=' ,null);
                }else{
                    $query->where('instrument_mode', $request->instrument_mode);
                }
            }
        }

        $collections = $query->orderByRaw('CAST(ref_no AS UNSIGNED) DESC')->paginate($perPage)->appends($request->except('page'));
        $customers = Customer::select('id','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('collection.index', compact('collections', 'perPage','customers'));
    }

    public function collEntry()
    {
        if (!auth()->user()->can('Create Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $banks = CustomerBank::orderBy('bank_name')->get();
        $depBanks = [];
        $depCash = [];

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        // Bank GL Code
        $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
        if($getBankHead){
            $glCodeBank = $getBankHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeBank,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depBanks = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
                if($e->getMessage()){
                    return redirect()->route('dashboard')->with('error', 'API Connection Timeout');
                }
            }
        }

        // Cash GL Code
        $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
        if($getCashHead){
            $glCodeCash = $getCashHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeCash,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depCash = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }

        //get department list from api - salaquzzaman @ 05-06-2024
        $departments = [];
        try{
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        }catch (\Exception $e) {
            // Log::error($e->getMessage());
        }

        $customers = Customer::select('autoAcNo','customer_name')
            ->where('status','Active')
            ->orderBy('customer_name', 'asc')
            ->get();



        return view('collection.entry', compact('customers','banks','depBanks','depCash','departments'));
    }

    public function collSearch($customerId=null){

        $bills = [];
        if($customerId && ($fndCustomer = Customer::where('autoAcNo',$customerId)->first())){
            $bills = DB::select("SELECT * FROM bill_processes
                    WHERE customer_id = $fndCustomer->id
                    AND delete_flag=0
                    AND (payment_status != 'Paid' || payment_status = '' || payment_status IS NULL)");
        }
        echo view('collection.result', compact('bills'));
    }

    public function userAdvanceAmount($customerId){
        $customerInfo = Customer::where('autoAcNo',$customerId)->first();
        $advAmount = AdvancedCollection::where('customer',$customerInfo->id)->sum('amount');
        $debitAmount = CustomerDebit::where('customer_id',$customerInfo->id)->where('delete_flag',0)->sum('amount');
        echo $advAmount - $debitAmount;
    }

    public function store(Request $request)
    {

        if (!auth()->user()->can('Create Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $validator = Validator::make($request->all(), [
            'total_coll_amount' => 'required',
        ]);
        if ($validator->fails()) {
            return redirect()->route('coll.entry')
                         ->with('error', 'Missing required fields.');
        }

        $totalColAmt = $request->total_coll_amount;
        if(ceil($totalColAmt)<1){
            return redirect()->route('coll.entry')
                         ->with('error', 'Missing collection amount.');
        }

        $refSLNo = Collection::max('sl_no')+1;
        $refNno = sprintf('%04d%05d', date('ym'), $refSLNo);

        $fndCustomer = Customer::where('autoAcNo',$request->customer_id)->first();
        $custPKID = $fndCustomer->id;

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $depDivCode = '';
        $depDivName = '';
        $cashDivCode = '';
        $cashDivName = '';

        if($request->instrument_mode=='Cheque' || $request->instrument_mode=='BEFTN' || $request->instrument_mode=='Pay Order'){

            // Bank GL Code
            // $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
            // if($getBankHead){
            //     $glCodeBank = $getBankHead->gl_code;

            //     //get bank list from api - salaquzzaman @ 01-06-2024
            //     try{
            //         $response = Http::withHeaders([
            //             'Authorization' => 'Bearer ' . $token,
            //         ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
            //             'groupCode' => $glCodeBank,
            //             'companyNo' => $company_number,
            //         ]);

            //         if ($response->successful()) {
            //             $responseData = $response->json();
            //             if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
            //                 $depBanks = json_encode($responseData['accHeadDetailsInfos']);

            //                 // Decode JSON data to PHP array
            //                 $accounts = json_decode($depBanks, true);
            //                 $result = null;

            //                 // Search for the specific account by autoAcNo
            //                 foreach ($accounts as $account) {
            //                     if ($account['autoAcNo'] == $request->dep_bank_id) {
            //                         $result = $account;
            //                         break;
            //                     }
            //                 }
            //                 $depDivCode = $result['divCode'];
            //                 $depDivName = $result['divName'];
            //             }
            //         }
            //     }catch (\Exception $e) {
            //         // Log::error($e->getMessage());
            //     }
            // }

            $depDivCode = $request->department_id;
            $depDivName = $request->depDivName;

        }else if($request->instrument_mode=='Cash'){

            // CASH GL Code
            // $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
            // if($getCashHead){
            //     $glCodeCash = $getCashHead->gl_code;

            //     //get cash list from api - salaquzzaman @ 22-08-2024
            //     try{
            //         $response = Http::withHeaders([
            //             'Authorization' => 'Bearer ' . $token,
            //         ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
            //             'groupCode' => $glCodeCash,
            //             'companyNo' => $company_number,
            //         ]);

            //         if ($response->successful()) {
            //             $responseData = $response->json();
            //             if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
            //                 $depCash = json_encode($responseData['accHeadDetailsInfos']);

            //                 // Decode JSON data to PHP array
            //                 $accounts = json_decode($depCash, true);
            //                 $result = null;

            //                 // Search for the specific account by autoAcNo
            //                 foreach ($accounts as $account) {
            //                     if ($account['autoAcNo'] == $request->dep_cash_id) {
            //                         $result = $account;
            //                         break;
            //                     }
            //                 }
            //                 $cashDivCode = $result['divCode'];
            //                 $cashDivName = $result['divName'];
            //             }
            //         }
            //     }catch (\Exception $e) {
            //         // Log::error($e->getMessage());
            //     }
            // }


            $cashDivCode = $request->department_id;
            $cashDivName = $request->depDivName;

        }else{

        }

        $cusDivCode = '';
        $cusDivName = '';

        // Customer GL Code
        // $getCusHead = GlHead::where('gl_subject',strtoupper('Accounts Receivable'))->first();
        // if($getCusHead){
        //     $glCodeCus = $getCusHead->gl_code;

        //     //get customer list from api - salaquzzaman @ 03-06-2024
        //     try{
        //         $response2 = Http::withHeaders([
        //             'Authorization' => 'Bearer ' . $token,
        //         ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
        //             'groupCode' => $glCodeCus,
        //             'companyNo' => $company_number,
        //         ]);

        //         if ($response2->successful()) {
        //             $responseData2 = $response2->json();
        //             if ($responseData2 && isset($responseData2['accHeadDetailsInfos'])) {
        //                 $customers = json_encode($responseData2['accHeadDetailsInfos']);

        //                 // Decode JSON data to PHP array
        //                 $accounts = json_decode($customers, true);
        //                 $result = null;

        //                 // Search for the specific account by autoAcNo
        //                 foreach ($accounts as $account) {
        //                     if ($account['autoAcNo'] == $fndCustomer->autoAcNo) {
        //                         $result = $account;
        //                         break;
        //                     }
        //                 }
        //                 $cusDivCode = $result['divCode'];
        //                 $cusDivName = $result['divName'];
        //             }
        //         }
        //     }catch (\Exception $e) {
        //         // Log::error($e->getMessage());
        //     }
        // }


        $cusDivCode = $request->department_id;
        $cusDivName = $request->depDivName;

        $taxAutoAcNo = '';
        $taxDivCode = '';
        $taxDivName = '';

        if(round($request->tds_amount)>0){

            // TDS/TAX Amount
            $getTaxHead = GlHead::where('gl_subject',strtoupper('TAX'))->first();
            if($getTaxHead){
                $glCodeTax = $getTaxHead->gl_code;

                //get tds/tax code details from api - salaquzzaman @ 08-06-2024
                try{
                    $response3 = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                        'groupCode' => $glCodeTax,
                        'companyNo' => $company_number,
                    ]);

                    if ($response3->successful()) {
                        $responseData3 = $response3->json();
                        if ($responseData3 && isset($responseData3['accHeadDetailsInfo'])) {

                            $taxAutoAcNo = $responseData3['accHeadDetailsInfo']['autoAcNo'];
                            // $taxDivCode = $responseData3['accHeadDetailsInfo']['divCode'];
                            // $taxDivName = $responseData3['accHeadDetailsInfo']['divName'];
                            $taxDivCode = $request->department_id;
                            $taxDivName = $request->depDivName;
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }


        $vatAutoAcNo = '';
        $vatDivCode = '';
        $vatDivName = '';

        if(round($request->vat_amount)>0){

            // TDS/TAX Amount
            $getVatHead = GlHead::where('gl_subject',strtoupper('VAT'))->first();
            if($getVatHead){
                $glCodeVat = $getVatHead->gl_code;

                //get vat code details from api - salaquzzaman @ 08-06-2024
                try{
                    $response4 = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                        'groupCode' => $glCodeVat,
                        'companyNo' => $company_number,
                    ]);

                    if ($response4->successful()) {
                        $responseData4 = $response4->json();
                        if ($responseData4 && isset($responseData4['accHeadDetailsInfo'])) {

                            $vatAutoAcNo = $responseData4['accHeadDetailsInfo']['autoAcNo'];
                            // $vatDivCode = $responseData4['accHeadDetailsInfo']['divCode'];
                            // $vatDivName = $responseData4['accHeadDetailsInfo']['divName'];
                            $vatDivCode = $request->department_id;
                            $vatDivName = $request->depDivName;
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }


        $adjAutoAcNo = '';
        $adjDivCode = '';
        $adjDivName = '';

        if(round($request->adjust_amount)>0){

            // TDS/TAX Amount
            $getAdjHead = GlHead::where('gl_subject',strtoupper('Provission For Bad Debts'))->first();
            if($getAdjHead){
                $glCodeAdj = $getAdjHead->gl_code;

                //get adjust details from api - salaquzzaman @ 08-06-2024
                try{
                    $response5 = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                        'groupCode' => $glCodeAdj,
                        'companyNo' => $company_number,
                    ]);

                    if ($response5->successful()) {
                        $responseData5 = $response5->json();
                        if ($responseData5 && isset($responseData5['accHeadDetailsInfo'])) {

                            $adjAutoAcNo = $responseData5['accHeadDetailsInfo']['autoAcNo'];
                            // $adjDivCode = $responseData5['accHeadDetailsInfo']['divCode'];
                            // $adjDivName = $responseData5['accHeadDetailsInfo']['divName'];
                            $adjDivCode = $request->department_id;
                            $adjDivName = $request->depDivName;
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }

        ($taxAutoAcNo)? $TTX_NA = '||Collection from '.$fndCustomer->customer_name : $TTX_NA = '';
        ($taxAutoAcNo)? $TTX_A = ','.$taxAutoAcNo : $TTX_A = '';
        ($taxAutoAcNo)? $TTX_D = ','.$request->tds_amount : $TTX_D = '';
        ($taxAutoAcNo)? $TTX_C = ',0' : $TTX_C = '';
        ($taxAutoAcNo)? $TTX_N = '||'.$taxDivName : $TTX_N = '';
        ($taxAutoAcNo)? $TTX_CO = ','.$taxDivCode : $TTX_CO = '';

        ($vatAutoAcNo)? $VAAT_NA = '||Collection from '.$fndCustomer->customer_name : $VAAT_NA = '';
        ($vatAutoAcNo)? $VAAT_A = ','.$vatAutoAcNo : $VAAT_A = '';
        ($vatAutoAcNo)? $VAAT_D = ','.$request->vat_amount : $VAAT_D = '';
        ($vatAutoAcNo)? $VAAT_C = ',0' : $VAAT_C = '';
        ($vatAutoAcNo)? $VAAT_N = '||'.$vatDivName : $VAAT_N = '';
        ($vatAutoAcNo)? $VAAT_CO = ','.$vatDivCode : $VAAT_CO = '';

        ($adjAutoAcNo)? $ADJ_NA = '||Collection from '.$fndCustomer->customer_name : $ADJ_NA = '';
        ($adjAutoAcNo)? $ADJ_A = ','.$adjAutoAcNo : $ADJ_A = '';
        ($adjAutoAcNo)? $ADJ_D = ','.$request->adjust_amount : $ADJ_D = '';
        ($adjAutoAcNo)? $ADJ_C = ',0' : $ADJ_C = '';
        ($adjAutoAcNo)? $ADJ_N = '||'.$adjDivName : $ADJ_N = '';
        ($adjAutoAcNo)? $ADJ_CO = ','.$adjDivCode : $ADJ_CO = '';

        $rcvAmt = (float)($request->received_amount);
        $tdsAmt = (float)($request->tds_amount);
        $vatAmt = (float)($request->vat_amount);
        $adjAmt = (float)($request->adjust_amount);

        //$advance_amount = (float)$request->advance_amount;
        $deducted_amount = (float)$request->deducted_amount;
        $total_coll_amount = (float)$request->total_coll_amount;

        $voucherMode = 'JV';
        if($request->instrument_mode=='Cheque' || $request->instrument_mode=='BEFTN'){
            $voucherMode = 'BRV';
        }else if($request->instrument_mode=='Cash'){
            $voucherMode = 'CRV';
        }else{}

        // Advance adjustment
        if($deducted_amount>0){
            $advCustomer = AdvancedCustomer::where('customerhead_id',$fndCustomer->id)->first();
            if(!$advCustomer){
                return redirect()->route('coll.entry')
                             ->with('error', 'Advanced customer setup not found.');
            }

            $ADV_NA = "||Collection from $fndCustomer->customer_name";
            $ADV_A = ','.$advCustomer->customeradvancedhead_autoAcNo;
            $ADV_D = ','.$deducted_amount;
            $ADV_C = ',0';
            $ADV_N = '||'.$request->depDivName;
            $ADV_CO = ','.$request->department_id;
        }else{
            $ADV_NA = '';
            $ADV_A = '';
            $ADV_D = '';
            $ADV_C = '';
            $ADV_N = '';
            $ADV_CO = '';
        }

        $remaining_amount = 0;
        $remaining_amount = $total_coll_amount - ($deducted_amount + ($tdsAmt+$vatAmt+$adjAmt));

        if($request->instrument_mode=='Cheque' || $request->instrument_mode=='BEFTN' || $request->instrument_mode=='Pay Order'){
            $BANK_NA = "||Collection from $fndCustomer->customer_name";
            $BANK_A = ','.$request->dep_bank_id;
            $BANK_D = ','.$remaining_amount;
            $BANK_C = ',0';
            $BANK_N = '||'.$depDivName;
            $BANK_CO = ','.$depDivCode;
        }else if($request->instrument_mode=='Cash'){
            $BANK_NA = "||Collection from $fndCustomer->customer_name";
            $BANK_A = ','.$request->dep_cash_id;
            $BANK_D = ','.$remaining_amount;
            $BANK_C = ',0';
            $BANK_N = '||'.$cashDivName;
            $BANK_CO = ','.$cashDivCode;
        }else{
            $BANK_NA = '';
            $BANK_A = '';
            $BANK_D = '';
            $BANK_C = '';
            $BANK_N = '';
            $BANK_CO = '';
        }

        $customerAMT = 0;
        if($rcvAmt<1 && $tdsAmt<1 && $vatAmt<1 && $adjAmt>0){
            // only adjust amount/ bad depts
            $BANK_NA = '';
            $BANK_A = '';
            $BANK_D = '';
            $BANK_C = '';
            $BANK_N = '';
            $BANK_CO = '';
        }

        // upload cheque/beftn attachemnt
        $attachFileName = null;
        if($request->instrument_mode=='Cheque' || $request->instrument_mode=='BEFTN' || $request->instrument_mode=='Pay Order'){
            if ($request->hasFile('attachment')) {
                $file = $request->file('attachment');
                $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
                $file->storeAs ('public/vds-certificate', $filename);
                $attachFileName = $filename;
            }
        }

        // upload tds certificate
        $tdsFileName = null;
        if($request->set_tds==1){
            if ($request->hasFile('tds_certificate')) {
                $file = $request->file('tds_certificate');
                $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
                $file->storeAs ('public/vds-certificate', $filename);
                $tdsFileName = $filename;
            }
        }

        // upload vds certificate
        $vdsFileName = null;
        if($request->set_vds==1){
            if ($request->hasFile('vds_certificate')) {
                $file = $request->file('vds_certificate');
                $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
                $file->storeAs ('public/vds-certificate', $filename);
                $vdsFileName = $filename;
            }
        }


        $narration = "Collection from $fndCustomer->customer_name$BANK_NA$TTX_NA$VAAT_NA$ADJ_NA$ADV_NA";
        $autoAcNo = $fndCustomer->autoAcNo.$BANK_A.$TTX_A.$VAAT_A.$ADJ_A.$ADV_A;
        $debitAmt = "$customerAMT$BANK_D$TTX_D$VAAT_D$ADJ_D$ADV_D";
        $creditAmt = $total_coll_amount.$BANK_C.$TTX_C.$VAAT_C.$ADJ_C.$ADV_C;
        $divName = "$cusDivName$BANK_N$TTX_N$VAAT_N$ADJ_N$ADV_N";
        $divCode = "$cusDivCode$BANK_CO$TTX_CO$VAAT_CO$ADJ_CO$ADV_CO";

        $postingDate = Carbon::createFromFormat('d/m/Y', $request->bank_deposit_date)->format('Y-m-d');

        try{

            $instru_mode = $request->instrument_mode;

            $collection                   = new Collection;
            $collection->sl_no            = $refSLNo;
            $collection->ref_no           = $refNno;
            $collection->customer_id      = $custPKID;

            $collection->customerAutoAc   = $fndCustomer->autoAcNo;
            $collection->tdsAutoAc        = str_replace(',', '', $TTX_A);
            $collection->vatAutoAc        = str_replace(',', '', $VAAT_A);
            $collection->adjustAutoAc     = str_replace(',', '', $ADJ_A);
            $collection->advanceAutoAc    = str_replace(',', '', $ADV_A);
            $collection->receive_tds      = $request->set_tds;
            $collection->receive_vds      = $request->set_vds;
            $collection->tds_certificate  = $tdsFileName;
            $collection->vds_certificate  = $vdsFileName;
            $collection->attachment       = $attachFileName;

            $collection->coll_date        = date('d/m/Y');
            $collection->instrument_mode  = $instru_mode;
            $collection->instrument_no    = $request->instrument_no;
            $collection->instrument_date  = $request->instrument_date;
            $collection->bank_id          = $request->bank_id;
            $collection->received_amount  = $request->received_amount;
            $collection->tds_amount       = $request->tds_amount;
            $collection->vat_amount       = $request->vat_amount;
            $collection->adjust_amount    = $request->adjust_amount;
            $collection->total_coll_amount   = $request->total_coll_amount;
            $collection->advance_amount       = $request->deducted_amount;
            $collection->money_receipt_no    = $request->money_receipt_no;
            $collection->money_receipt_date  = $request->money_receipt_date;
            $collection->bank_slip_no        = $request->bank_slip_no;
            $collection->bank_deposit_date   = $request->bank_deposit_date;
            $collection->dep_bank_id         = ($instru_mode=='Cheque' || $instru_mode=='BEFTN' || $instru_mode=='Pay Order')? $request->dep_bank_id:$request->dep_cash_id;
            $collection->account_name        = ($instru_mode!='')? $request->account_name:null;
            $collection->adj_remarks         = $request->adj_remarks;
            $collection->uuid                = mt_rand(111111,999999);
            $collection->created_by          = Auth::user()->id;

            if($collection->save()){

                $billCollectionId = $collection->id.$collection->uuid;

                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->post($api_ip.'/api/v1/voucherEntry/saveVoucherInfo', [
                    'voucherDt' => $postingDate,
                    'voucherFlag' => $voucherMode,
                    'narration' =>  $narration,
                    'verifyFlag' => 'N',
                    'autoAcNo' => $autoAcNo,
                    'debitAmt' => $debitAmt,
                    'creditAmt' => $creditAmt,
                    'companyNo' => $company_number,
                    'divName' => $divName,
                    'divCode' => $divCode,
                    'chqNo' => ($request->instrument_mode=='Cheque' || $request->instrument_mode=='BEFTN')? $request->bank_slip_no:'',
                    'chqDt' => ($request->instrument_mode=='Cheque' || $request->instrument_mode=='BEFTN')? $postingDate:'',
                    'erpTranType' => 'billCollection',
                    'erpTranId' => $billCollectionId,
                ]);
                if($response->successful()){
                    // store to db

                    $billId = $request->billId;
                    if($billId){
                        $k = 0;
                        foreach($billId as $blID){

                            // Bill Process Update
                            $bill = BillProcess::find($blID);
                            $bill->payment_status     = $request->flagStatus[$k];
                            $bill->due                = $request->remainingAmt[$k];
                            $bill->save();

                            // Collection Amount Insert
                            if(floatval($request->collectionAmt[$k])>0){
                                $colChd                    = new CollectionChd;
                                $colChd->collection_id     = $collection->id;
                                $colChd->ref_no            = $collection->ref_no;
                                $colChd->bill_process_id   = $blID;
                                $colChd->project_id        = null;
                                $colChd->invoice_no        = $bill->invoice_no;
                                $colChd->collection_amount = $request->collectionAmt[$k];
                                $colChd->collection_date   = $collection->coll_date;
                                $colChd->save();
                            }
                            $k = $k + 1;
                        }
                    }

                    if($deducted_amount>0){
                        $debit                  = new CustomerDebit;
                        $debit->collection_id   = $collection->id;
                        $debit->customer_id     = $custPKID;
                        $debit->amount          = $deducted_amount;
                        $debit->coll_date       = $collection->coll_date;
                        $debit->posting_date    = $request->bank_deposit_date;
                        $debit->save();

                        // Customer Ledger
                        // $ledger                 = new CustomerLedger();
                        // $ledger->entry_date     = date('Y-m-d');
                        // $ledger->narration      = 'Advance Collection';
                        // $ledger->customer_id    = $collection->customer_id;
                        // $ledger->type           = 'Advance-collection';
                        // $ledger->ref_id         = $collection->id;
                        // $ledger->ref_no         = $collection->ref_no;
                        // $ledger->amount         = $deducted_amount;
                        // $ledger->created_by     = auth()->id();
                        // $ledger->save();
                    }

                    $withoutDeducted = $collection->total_coll_amount - $deducted_amount;
                    if($withoutDeducted>0){
                        // Customer Ledger
                        // $ledger                 = new CustomerLedger();
                        // $ledger->entry_date     = date('Y-m-d');
                        // $ledger->narration      = 'Collection';
                        // $ledger->customer_id    = $collection->customer_id;
                        // $ledger->type           = 'Collection';
                        // $ledger->ref_id         = $collection->id;
                        // $ledger->ref_no         = $collection->ref_no;
                        // $ledger->amount         = -$withoutDeducted;
                        // $ledger->created_by     = auth()->id();
                        // $ledger->save();
                    }


                    return redirect()->route('collection.index')
                            ->with('success', 'Collection successfull. Ref No: '.$refNno);

                }else{
                    $collection->delete();
                    return redirect()->route('coll.entry')
                            ->with('error', 'Collection not received.');
                }
            }else{
                return redirect()->route('coll.entry')
                        ->with('error', 'Collection not received.');
            }

        }catch (\Exception $e) {
            if($e->getMessage()){
                return redirect()->route('collection.index')
                        ->with('error', 'Collection not received.');
            }
        }

    }

    public function show($collID)
    {
        if (!auth()->user()->can('View Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $udAcName = '';
        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $pay = Collection::find($collID);
        if($pay->instrument_mode=='Cheque' || $pay->instrument_mode=='BEFTN' || $pay->instrument_mode=='Pay Order'){

            // Bank GL Code
            $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
            if($getBankHead){
                $glCodeBank = $getBankHead->gl_code;

                try{
                    $response = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                        'groupCode' => $glCodeBank,
                        'companyNo' => $company_number,
                    ]);

                    if ($response->successful()) {
                        $responseData = $response->json();
                        if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                            $depBanks = $responseData['accHeadDetailsInfos'];
                            foreach ($depBanks as $row) {
                                if ($row['autoAcNo'] == $pay->dep_bank_id) {
                                    $udAcName = $row['udAcName'];
                                    break;
                                }
                            }
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }else{ //cash

            // CASH GL Code
            $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
            if($getCashHead){
                $glCodeCash = $getCashHead->gl_code;

                //get cash list from api - salaquzzaman @ 22-08-2024
                try{
                    $response = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                        'groupCode' => $glCodeCash,
                        'companyNo' => $company_number,
                    ]);

                    if ($response->successful()) {
                        $responseData = $response->json();
                        if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                            $depBanks = $responseData['accHeadDetailsInfos'];
                            foreach ($depBanks as $row) {
                                if ($row['autoAcNo'] == $pay->dep_bank_id) {
                                    $udAcName = $row['udAcName'];
                                    break;
                                }
                            }
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }

        // $payChd = CollectionChd::where('collection_id',$collID)->get();
        return view('collection.show', compact('pay','udAcName'));
    }

    private function normalizeDate($date)
    {
        $formats = [
            'd/m/y',
            'd/m/Y',
            'd-M-Y',
            'd-M-y'
        ];

        foreach ($formats as $format) {
            try {
                $parsedDate = \Carbon\Carbon::createFromFormat($format, $date);
                return $parsedDate->format('d/m/Y');
            } catch (\Exception $e) {
            }
        }

        return $date;
    }




    // public function advancedCollection()
    // {
    //     $advancedcollections = AdvancedCollection::all();

    //     $customerIds = $advancedcollections->pluck('customer')->unique()->toArray();
    //     $customerNames = DB::table('billing_customers')->whereIn('id', $customerIds)->pluck('customer_name', 'id')->toArray();

    //     $bankIds = $advancedcollections->pluck('bank')->unique()->toArray();
    //     $bankNames = DB::table('depository_banks')->whereIn('id', $bankIds)->pluck('bank_name', 'id')->toArray();

    //     return view('collection.advancedcollectionindex', compact('advancedcollections', 'customerNames', 'bankNames'));
    // }



    public function advancedCollection(Request $request)
    {
        $perPage = $request->input('per_page', 25);

        $advancedcollections = AdvancedCollection::paginate($perPage);

        $customerIds = $advancedcollections->pluck('customer')->unique()->toArray();
        $customerNames = DB::table('billing_customers')->whereIn('id', $customerIds)->pluck('customer_name', 'id')->toArray();

        $bankIds = $advancedcollections->pluck('bank')->unique()->toArray();
        $bankNames = DB::table('depository_banks')->whereIn('id', $bankIds)->pluck('bank_name', 'id')->toArray();

        return view('collection.advancedcollectionindex', compact('advancedcollections', 'customerNames', 'bankNames', 'perPage'));
    }




    // Advanced Collection Option
    public function advancedcollectionCreate()
    {

        //$customers = Customer::where('status','Active')->orderBy('customer_name', 'asc')->get();

        $distinctCustomerIds = AdvancedCustomer::select('customerhead_id')->distinct()->pluck('customerhead_id');
        $customers = Customer::whereIn('id', $distinctCustomerIds)
            ->select('autoAcNo','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $banks = CustomerBank::orderBy('bank_name')->get();
        $depBanks = [];
        $depCash = [];

        // Bank GL Code
        $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
        if($getBankHead){
            $glCodeBank = $getBankHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeBank,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depBanks = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }

        // Cash GL Code
        $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
        if($getCashHead){
            $glCodeCash = $getCashHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeCash,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depCash = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }


        //get department list from api - salaquzzaman @ 05-06-2024
        $departments = [];
        try{
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        }catch (\Exception $e) {
            // Log::error($e->getMessage());
        }

        return view('collection.advancedcollectioncreate', compact('customers', 'banks', 'depBanks','departments','depCash'));
    }



    public function advancedcollectionStore(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'amount' => 'required|numeric|min:1',

        ]);

        if ($validator->fails()) {
            return redirect()->route('advanced.collection.create')
                ->with('error', 'Missing required fields.');
        }

        // Check if customer exists
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        }

        $advData = AdvancedCustomer::where('customerhead_id',$custPKID)->first();
        if(!$advData){
            return redirect()->route('advanced.collection.create')
                ->with('error', 'Please setup customer against advance customer setup.');
        }

        $bankPKID = null;
        if($request->instrument_mode=='Bank'){

            $fndBank = DepositoryBank::where('bank_id', $request->dep_bank_id)->first();
            if ($fndBank) {
                $bankPKID = $fndBank->id;
                $bankAutoAcNo = $fndBank->bank_id;
            } else {
                // Create new bank
                $bank_name = $request->bankName;
                $autoAcNo = $request->dep_bank_id;
                $phoneNo = '0' . mt_rand(11111, 99999);
                $holder = 'User-' . mt_rand(11111, 999999);
                $bankInfo = [
                    'holder_name' => $holder,
                    'bank_name' => $bank_name,
                    'account_number' => 0,
                    'opening_balance' => 0,
                    'phone' => $phoneNo,
                    'bank_id' => $autoAcNo,
                    'divCode' => $request->bankDivCode,
                    'created_by' => \auth()->id(),
                    'is_active' => 1
                ];

                // Log::info('Creating new bank:', $bankInfo); // Log the bank info

                $bankNew = DepositoryBank::create($bankInfo);
                $bankPKID = $bankNew->id;
                $bankAutoAcNo = $bankNew->bank_id;
            }
        }

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $setAutoAC = ($request->instrument_mode=='Bank')? $bankAutoAcNo:$request->dep_cash_id;
        $customerAgnstAC = $advData->customeradvancedhead_autoAcNo;

        $narration = 'Advance Collection||Advance Collection';
        $autoAcNo = $setAutoAC.','.$customerAgnstAC;
        $debitAmt = $request->amount.',0';
        $creditAmt = '0,'.$request->amount;
        $divName = "$request->depDivName||$request->depDivName";
        $divCode = "$request->department_id,$request->department_id";

        try{
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->post($api_ip.'/api/v1/voucherEntry/saveVoucherInfo', [
                'voucherDt' => date('Y-m-d'),
                'voucherFlag' => 'JV',
                'narration' =>  $narration,
                'verifyFlag' => 'N',
                'autoAcNo' => $autoAcNo,
                'debitAmt' => $debitAmt,
                'creditAmt' => $creditAmt,
                'companyNo' => $company_number,
                'divName' => $divName,
                'divCode' => $divCode
            ]);
            if($response->successful()){

                $attachment = null;
                if ($request->hasFile('attachment')) {
                    $file = $request->file('attachment');
                    $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
                    $file->storeAs ('public/vds-certificate', $filename);
                    $attachment = $filename;
                }

                // store to db
                $adv = AdvancedCollection::create([
                    'customer' => $custPKID,
                    'amount' => $request->amount,
                    'instrument_mode' => $request->instrument_mode,
                    'instrument_no'   => $request->instrument_no,
                    'instrument_date' => $request->instrument_date,
                    'attachment'      => $attachment,
                    'department_id' => $request->department_id,
                    'depDivName' => $request->depDivName,
                    'bank' => ($request->instrument_mode=='Bank')? $bankPKID:$request->dep_cash_id
                ]);

                // Customer Ledger
                $ledger                 = new CustomerLedger();
                $ledger->entry_date     = date('Y-m-d');
                $ledger->narration      = 'Customer Advance';
                $ledger->customer_id    = $custPKID;
                $ledger->type           = 'Advance';
                $ledger->ref_id         = $adv->id;
                $ledger->ref_no         = $adv->id;
                $ledger->amount         = -($request->amount);
                $ledger->created_by     = auth()->id();
                $ledger->save();

                return redirect()->route('advanced.collection')->with('success', 'Advanced collection created successfully.');


            }else{
                return redirect()->route('advanced.collection')
                        ->with('error', 'Collection not created.');
            }

        }catch (\Exception $e) {
            if($e->getMessage()){
                return redirect()->route('advanced.collection')
                        ->with('error', 'Collection not created.');
            }
        }

    }

    public function advancedcollectionView($id)
    {
        $advancedcollection = AdvancedCollection::findOrFail($id);

        $customerIds = collect([$advancedcollection->customer]);
        $customerNames = DB::table('billing_customers')->whereIn('id', $customerIds)->pluck('customer_name', 'id')->toArray();

        $bankIds = collect([$advancedcollection->bank]);
        $bankNames = DB::table('depository_banks')->whereIn('id', $bankIds)->pluck('bank_name', 'id')->toArray();

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $depCash = [];

        // Cash GL Code
        $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
        if($getCashHead){
            $glCodeCash = $getCashHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeCash,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depCash = json_encode($responseData['accHeadDetailsInfos']);
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }

        return view('collection.advancedcollectionview', compact('advancedcollection', 'customerNames', 'bankNames','depCash'));
    }



public function advancedcollectionEdit($id)
{
    $collection = AdvancedCollection::findOrFail($id);

    $customers = [];
    $token = session('api_token');
    $api_ip = env("API_IP");
    $company_number = env("COMPANY_NUMBER");

    $getCusHead = GlHead::where('gl_subject', strtoupper('Accounts Receivable'))->first();
    if ($getCusHead) {
        $glCodeCus = $getCusHead->gl_code;

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip . '/api/v1/integrate/searchByGroupCode', [
                'groupCode' => $glCodeCus,
                'companyNo' => $company_number,
            ]);

            if ($response->successful()) {
                $responseData = $response->json();
                if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                    $customers = $responseData['accHeadDetailsInfos'];
                }
            }
        } catch (\Exception $e) {
        }
    }

    $autoAcNo = '';
    if($collection->customer && ($custInfo = Customer::find($collection->customer))){

        $autoAcNo = $custInfo->autoAcNo;
    }



    $depBanks = [];
    $getBankHead = GlHead::where('gl_subject', strtoupper('Bank'))->first();
    if ($getBankHead) {
        $glCodeBank = $getBankHead->gl_code;

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip . '/api/v1/integrate/searchByGroupCode', [
                'groupCode' => $glCodeBank,
                'companyNo' => $company_number,
            ]);

            if ($response->successful()) {
                $responseData = $response->json();
                if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                    $depBanks = $responseData['accHeadDetailsInfos'];
                }
            }
        } catch (\Exception $e) {

        }
    }


    $bankautoAcNo = '';

    if($collection->bank && ($banksInfo = DepositoryBank::find($collection->bank))){
        $bankautoAcNo = $banksInfo->bank_id;
    }

    return view('collection.advancedcollectionedit', compact('customers', 'depBanks', 'collection', 'autoAcNo', 'bankautoAcNo'));
}



    public function advancedcollectionUpdate(Request $request, $id)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'amount' => 'required|numeric|min:1',
            'dep_bank_id' => 'required',
        ]);

        if ($validator->fails()) {
            return redirect()->route('advanced.collection.edit', $id)
                ->with('error', 'Missing required fields.')
                ->withErrors($validator)
                ->withInput();
        }

        // Check if customer exists
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        } else {
            // Create new customer
            $customer_name = $request->customerName;
            $email = Settings::genRandString(6) . '@example.com';
            $autoAcNo = $request->customer_id;
            $phoneNo = '0' . mt_rand(11111, 999999);
            $cusInfo = [
                'customer_name' => $customer_name,
                'phone' => $phoneNo,
                'billing_name' => $customer_name,
                'billing_phone' => $phoneNo,
                'billing_email' => $email,
                'account_name' => $customer_name . ' A/C',
                'account_phone' => $phoneNo,
                'account_email' => $email,
                'status' => 'Active',
                'autoAcNo' => $autoAcNo,
            ];

            Log::info('Creating new customer:', $cusInfo); // Log the customer info

            $newCust = Customer::create($cusInfo + ['created_by' => auth()->id()]);
            $custPKID = $newCust->id;
        }

        // Check if bank exists
        $fndBank = DepositoryBank::where('bank_id', $request->dep_bank_id)->first();
        if ($fndBank) {
            $bankPKID = $fndBank->id;
            $bankAutoAcNo = $fndBank->bank_id;
        } else {
            // Create new bank
            $bank_name = $request->bankName;
            $autoAcNo = $request->dep_bank_id;
            $phoneNo = '0' . mt_rand(11111, 99999);
            $holder = 'User-' . mt_rand(11111, 999999);
            $bankInfo = [
                'holder_name' => $holder,
                'bank_name' => $bank_name,
                'account_number' => 0,
                'opening_balance' => 0,
                'phone' => $phoneNo,
                'bank_id' => $autoAcNo,
                'divCode' => $request->bankDivCode,
                'created_by' => \auth()->id(),
                'is_active' => 1
            ];

            Log::info('Creating new bank:', $bankInfo); // Log the bank info

            $bankNew = DepositoryBank::create($bankInfo);
            $bankPKID = $bankNew->id;
            $bankAutoAcNo = $bankNew->bank_id;
        }

        // Update the advanced collection
        $advancedCollection = AdvancedCollection::findOrFail($id);
        $advancedCollection->update([
            'customer' => $custPKID,
            'amount' => $request->amount,
            'bank' => $bankPKID,
        ]);

        return redirect()->route('advanced.collection')->with('success', 'Advanced collection updated successfully.');
    }


    public function advancedcollectionDelete($id)
    {
        $advancedcollections = AdvancedCollection::findOrFail($id);
        $advancedcollections->delete();

        return redirect()->route('advanced.collection', $advancedcollections) ->with('success', 'Advanced Collection deleted successfully');
    }


    public function vdsList(Request $request)
    {
        if (!auth()->user()->can('View Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 25);

        $query = Collection::query();

        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(bank_deposit_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(bank_deposit_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('type')) {
            $query->where('delete_flag', 0);

            if ($request->type == 1) { // submitted
                $query->where('tds_certificate', '!=', '')
                      ->where('vds_certificate', '!=', '');
            } else { // type == 2 not submitted
                $query->where(function($query) {
                    $query->where('tds_certificate', '=', null)
                          ->orWhere('vds_certificate', '=', null);
                });
            }
        }

        $collections = $query->orderBy('id', 'desc')->paginate($perPage)->appends($request->except('page'));

        return view('collection.vds-list', compact('collections', 'perPage'));
    }

    public function vdsUploadPage($id){
        $collection = Collection::find($id);
        return view('collection.vds-upload', compact('id','collection'));
    }
    public function vdsUpload(Request $request){
        $collection = Collection::find($request->collID);

        if($collection->receive_tds!=1){
            if ($request->hasFile('tds_certificate')) {
                $file = $request->file('tds_certificate');
                $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
                $file->storeAs ('public/vds-certificate', $filename);
                $tdsFileName = $filename;
                $collection->receive_tds = 1;
                $collection->tds_certificate = $tdsFileName;
            }
        }
        if($collection->receive_vds!=1){
            if ($request->hasFile('vds_certificate')) {
                $file = $request->file('vds_certificate');
                $filename = time() . mt_rand(1000,9999) . '.' . strtolower($file->getClientOriginalExtension());
                $file->storeAs ('public/vds-certificate', $filename);
                $vdsFileName = $filename;
                $collection->receive_vds = 1;
                $collection->vds_certificate = $vdsFileName;
            }
        }
        $collection->save();
        $successURL = $request->successURL;
        return redirect($successURL)->with('success', 'Certificate uploaded successfully.');
    }

    // Duplicate Invoice
    public function cancelCollPreview(Request $request)
    {
        $query = Collection::query();

        if ($request->filled('ref_no')) {
            $query->where('ref_no', $request->ref_no);
            $query->where('delete_flag', 0);
        }

        $pay = $query->first();

        $udAcName = '';
        $payChd = [];
        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        if($pay){
            if($pay->instrument_mode=='Cheque' || $pay->instrument_mode=='BEFTN'){

                // Bank GL Code
                $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
                if($getBankHead){
                    $glCodeBank = $getBankHead->gl_code;

                    //get deposit bank list from api - salaquzzaman @ 05-06-2024
                    try{
                        $response = Http::withHeaders([
                            'Authorization' => 'Bearer ' . $token,
                        ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                            'groupCode' => $glCodeBank,
                            'companyNo' => $company_number,
                        ]);

                        if ($response->successful()) {
                            $responseData = $response->json();
                            if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                                $depBanks = $responseData['accHeadDetailsInfos'];
                                foreach ($depBanks as $row) {
                                    if ($row['autoAcNo'] == $pay->dep_bank_id) {
                                        $udAcName = $row['udAcName'];
                                        break;
                                    }
                                }
                            }
                        }
                    }catch (\Exception $e) {
                        // Log::error($e->getMessage());
                    }
                }

            }else{ //cash

                // CASH GL Code
                $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
                if($getCashHead){
                    $glCodeCash = $getCashHead->gl_code;

                    //get cash list from api - salaquzzaman @ 22-08-2024
                    try{
                        $response = Http::withHeaders([
                            'Authorization' => 'Bearer ' . $token,
                        ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                            'groupCode' => $glCodeCash,
                            'companyNo' => $company_number,
                        ]);

                        if ($response->successful()) {
                            $responseData = $response->json();
                            if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                                $depBanks = $responseData['accHeadDetailsInfos'];
                                foreach ($depBanks as $row) {
                                    if ($row['autoAcNo'] == $pay->dep_bank_id) {
                                        $udAcName = $row['udAcName'];
                                        break;
                                    }
                                }
                            }
                        }
                    }catch (\Exception $e) {
                        // Log::error($e->getMessage());
                    }
                }

            }
            $payChd = CollectionChd::where('collection_id',$pay->id)->get();
        }


        return view('collection.collection-cancel-preview', [
            'udAcName' => $udAcName,
            'pay' => $pay,
            'payChd' => $payChd
        ]);

    }

    public function cancelCollDetails($id)
    {
        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");


        $insertLedger = false;
        $collection = Collection::find($id);
        $collectionChd = CollectionChd::where('collection_id',$id)->get();

        $erpTranId = $collection->id.$collection->uuid;

        try{

            $response4 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/voucherEntry/voucherInfo/billCollection/'.$erpTranId.'/'.$company_number);

            if ($response4->successful()) {

                $responseData4 = $response4->json();
                if(!$responseData4){
                    return redirect()->route('cancel.coll.preview')->with('error', 'Collection Cancellation not possible.');
                }
                if ($responseData4 && isset($responseData4['voucherMstId'])) {
                    $voucherMstId = $responseData4['voucherMstId'];
                    if($voucherMstId==''){
                        return redirect()->route('cancel.coll.preview')->with('error', 'Collection Cancellation not possible.');
                    }
                }

                if($voucherMstId){
                    // delete voucher procedure from api
                    $response = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->delete($api_ip.'/api/v1/deleteVoucher/'.$voucherMstId);

                    if($response->successful()){
                        $insertLedger = true;
                    }
                }

            }
        }catch (\Exception $e) {
            return redirect()->route('cancel.coll.preview')->with('error', 'Collection Cancellation not possible.');
        }

        if($insertLedger){

            // delete flag in collection table
            $collection->delete_flag = 1;
            $collection->save();

            // delete flag in collection chd table
            CollectionChd::where('collection_id',$id)->update(['delete_flag'=>'1']);

            // delete flag in customer debit/advance table
            if((float)($collection->advance_amount)>0){
                CustomerDebit::where('collection_id',$id)->update(['delete_flag'=>'1']);
            }

            foreach($collectionChd as $row){
                $billInfo = BillProcess::find($row->bill_process_id);
                $billAmount = (float)$billInfo->total_amount_vat;
                $newDue = (float)$billInfo->due + (float)$row->collection_amount;

                //previous collected amount
                $collectedInfo = CollectionChd::where('bill_process_id',$row->bill_process_id)
                    ->where('delete_flag',0)->sum('collection_amount');
                $collectedAmount = (float)$collectedInfo;

                if($collectedAmount==0){
                    $status = 'Unpaid';
                }else{
                    if($billAmount==$collectedAmount){
                        $status = 'Paid';
                    }else{
                        $status = 'Partial Paid';
                    }
                }
                $billInfo->due = $newDue;
                $billInfo->payment_status = $status;
                $billInfo->save();
            }

            $deducted_amount = (float)$collection->advance_amount;
            $withoutDeducted = $collection->total_coll_amount - $deducted_amount;

            if($deducted_amount>0){
                // Customer Ledger
                $ledger                 = new CustomerLedger();
                $ledger->entry_date     = date('Y-m-d');
                $ledger->narration      = 'Advance Collection Cancel';
                $ledger->customer_id    = $collection->customer_id;
                $ledger->type           = 'Advance-collection-cancel';
                $ledger->ref_id         = $collection->id;
                $ledger->ref_no         = $collection->ref_no;
                $ledger->amount         = -$deducted_amount;
                $ledger->created_by     = auth()->id();
                $ledger->save();
            }

            if($withoutDeducted>0){
                // Customer Ledger
                $ledger                 = new CustomerLedger();
                $ledger->entry_date     = date('Y-m-d');
                $ledger->narration      = 'Collection Cancel';
                $ledger->customer_id    = $collection->customer_id;
                $ledger->type           = 'Collection-cancel';
                $ledger->ref_id         = $collection->id;
                $ledger->ref_no         = $collection->ref_no;
                $ledger->amount         = $withoutDeducted;
                $ledger->created_by     = auth()->id();
                $ledger->save();
            }

            return redirect()->route('cancel.coll.preview')->with('success', 'Collection Cancelled successfully.');
        }else{
            return redirect()->route('cancel.coll.preview')->with('success', 'Collection Not Cancelled.');
        }








    }


    public function collMigrationStoreOld(){

        exit;

        // $collection_master_sil = 'aa_coll_mst_tgcl';
        // $collection_detail_sil = 'aa_coll_dtl_tgcl';
        // $bill_processes = 'aa_bill_processes';
        // $collections = 'aa_coll';
        // $collection_chds = 'aa_coll_chd';

        set_time_limit(0);

        $collection_master_sil = 'Collection_Master_SIL';
        $collection_detail_sil = 'Collection_Detail_SIL';
        $bill_processes = 'aa_bill_processes_test';
        $collections = 'aa_coll';
        $collection_chds = 'aa_coll_chd';

        $chds = DB::select("SELECT
            COLL_REF_NO,INVOICE_NO,COL_ADJ_AMT,COL_ADJ_DATE
            from $collection_detail_sil WHERE COLL_REF_NO IN (459,460,1521,1376)");
        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (3589,4095)");
        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (191,647,14446)");
        // $chds = DB::select("SELECT * from $collection_detail_sil");
        if($chds){
            $chdAMT = 0;
            foreach ($chds as $detail) {
                echo 'REF-NO: ';
                echo $COLL_REF_NO = $detail->COLL_REF_NO;
                echo '<br>';
                echo '......';
                echo $invNO = $detail->INVOICE_NO;
                echo '----';
                $singleAMT = $detail->COL_ADJ_AMT;
                $singleAMT = str_replace(",","",$singleAMT);
                echo 'Coll-INV-AMT: ';
                echo $singleAMT = $CollInvAmt = (float)str_replace(",","",$singleAMT);

                echo '----';
                // echo $detail->COL_ADJ_DATE;
                echo $COLL_REF_DATE = Carbon::createFromFormat('m/d/Y', $detail->COL_ADJ_DATE)->format('d/m/Y');
                echo '----';
                echo '<br>';

                // $chdAMT += (float)$singleAMT;

                if($invNO){
                    $invInfos = DB::select("SELECT customer_id,id,total_amount_vat from $bill_processes WHERE INVOICE_NO='$invNO'");
                    if($invInfos){
                        foreach($invInfos as $invDetail){

                            $collMstTGCL = DB::selectOne("SELECT
                                COLL_REF_DATE,ENTERED_DATE,MODIFIED_DATE,MR_DATE,COLL_MODE,BANK_ID,RECEIVED_AMT,TDS_AMT,VAT_AMT,ADJ_AMT,
                                TOT_COLL_AMT,MR_NO,ADJ_REMARKS,DS_NO,DS_BANK_ID,ENTERED_BY
                                from $collection_master_sil where COLL_REF_NO='$COLL_REF_NO'");
                            $COLL_REF_DATE_MST = Carbon::createFromFormat('m/d/Y', $collMstTGCL->COLL_REF_DATE)->format('d/m/Y');
                            $COLL_REF_DATE_MST_CREATE = Carbon::createFromFormat('d/m/Y', $COLL_REF_DATE_MST)->format('Y-m-d h:i:s');

                            $checkCollMst = DB::selectOne("SELECT id,ref_no from $collections where ref_no='$COLL_REF_NO'");
                            if($checkCollMst){
                                $collMstID = $checkCollMst->id;

                            }else{
                                // insert to collection
                                if($collMstTGCL->ENTERED_DATE!=''){
                                    $ENTERED_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->ENTERED_DATE)->format('Y-m-d h:i:s');
                                }else{
                                    $ENTERED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                if($collMstTGCL->MODIFIED_DATE!=''){
                                    $MODIFIED_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->MODIFIED_DATE)->format('Y-m-d h:i:s');
                                }else{
                                    $MODIFIED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                if($collMstTGCL->MR_DATE!=''){
                                    $MR_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->MR_DATE)->format('d/m/Y');
                                }else{
                                    $MR_DATE = null;
                                }
                                $collectionData = array(
                                    'ref_no' => $COLL_REF_NO,
                                    'customer_id' => $invDetail->customer_id,
                                    'coll_date' => $COLL_REF_DATE_MST,
                                    'bank_deposit_date' => $COLL_REF_DATE_MST,
                                    'instrument_mode' => $collMstTGCL->COLL_MODE,
                                    'bank_id' => $collMstTGCL->BANK_ID,
                                    'received_amount' => ((float)$collMstTGCL->RECEIVED_AMT>0)? $collMstTGCL->RECEIVED_AMT:0,
                                    'tds_amount' => ((float)$collMstTGCL->TDS_AMT>0)? $collMstTGCL->TDS_AMT:0,
                                    'vat_amount' => ((float)$collMstTGCL->VAT_AMT>0)? $collMstTGCL->VAT_AMT:0,
                                    'adjust_amount' => ((float)$collMstTGCL->ADJ_AMT>0)? $collMstTGCL->ADJ_AMT:0,
                                    'total_coll_amount' => ((float)$collMstTGCL->TOT_COLL_AMT>0)? $collMstTGCL->TOT_COLL_AMT:0,
                                    'advance_amount' => 0,
                                    'money_receipt_no' => $collMstTGCL->MR_NO,
                                    'money_receipt_date' => $MR_DATE,
                                    'adj_remarks' => $collMstTGCL->ADJ_REMARKS,
                                    'bank_slip_no' => $collMstTGCL->DS_NO,
                                    'dep_bank_id' => $collMstTGCL->DS_BANK_ID,
                                    'created_by' => $collMstTGCL->ENTERED_BY,
                                    'created_at' => $ENTERED_DATE,
                                    'updated_at' => $MODIFIED_DATE
                                );

                                DB::table($collections)->insert([$collectionData]);
                                $collMstID = DB::getPdo()->lastInsertId();
                            }

                            // echo '...............';
                            // echo 'Bill PK: ';
                            // echo $BillPK = $invDetail->id;
                            // echo '-----';
                            // echo 'Bill-Process-Amt: ';
                            // echo $BillProcessAmt = (float)$invDetail->total_amount_vat;
                            // echo '-----';

                            $BillPK = $invDetail->id;
                            $BillProcessAmt = (float)$invDetail->total_amount_vat;

                            $DueAmt = 0;
                            if($CollInvAmt >= $BillProcessAmt){
                                $CollChdAmt = $BillProcessAmt;
                                // $DueAmt = 0;
                                $CollInvAmt = $CollInvAmt - $BillProcessAmt;
                            }else{
                                $CollChdAmt = $CollInvAmt;
                                // $DueAmt = $BillProcessAmt - $CollInvAmt;
                                $CollInvAmt = 0;
                            }

                            if($CollChdAmt>0){
                                $chdData = array(
                                    'collection_id' => $collMstID,
                                    'ref_no' => $COLL_REF_NO,
                                    'bill_process_id' => $BillPK,
                                    'invoice_no' => $invNO,
                                    'collection_amount' => $CollChdAmt,
                                    'collection_date' => $COLL_REF_DATE_MST,
                                    'created_at' => $COLL_REF_DATE_MST_CREATE,
                                    'updated_at' => $COLL_REF_DATE_MST_CREATE
                                );

                                DB::table($collection_chds)->insert([$chdData]);
                            }

                            // $totalCollInfo = DB::selectOne("SELECT SUM(CAST(REPLACE(collection_amount, ',', '') AS DECIMAL(15, 2))) AS TotalColl
                            //     FROM $collection_chds WHERE bill_process_id = $BillPK");

                            // $DueAmt = $BillProcessAmt - $totalCollInfo->TotalColl;


                            // echo '----Remain-Coll-Inv-Amt: ';
                            // echo $CollInvAmt;
                            // echo '----Collection-Chd-Amt: ';
                            // echo $CollChdAmt;
                            // echo '----Due-Amt: ';
                            // echo $DueAmt;
                            // echo '<br>';

                        }
                    }
                }
            }
        }
    }


    public function collMigrationStoreOld2(){

        exit;

        // $collection_master_sil = 'aa_coll_mst_tgcl';
        // $collection_detail_sil = 'aa_coll_dtl_tgcl';
        // $bill_processes = 'aa_bill_processes';
        // $collections = 'aa_coll';
        // $collection_chds = 'aa_coll_chd';

        set_time_limit(0);

        $collection_master_sil = 'Collection_Master_SIL';
        $collection_detail_sil = 'Collection_Detail_SIL';
        $bill_processes = 'zzzz';
        $collections = 'zzzz_coll';
        $collection_chds = 'zzzz_coll_chd';

        $chds = DB::select("SELECT
            COLL_REF_NO,INVOICE_NO,COL_ADJ_AMT,COL_ADJ_DATE
            from $collection_detail_sil");
            // from $collection_detail_sil WHERE COLL_REF_NO IN (3837,4083)");
            // from $collection_detail_sil WHERE COLL_REF_NO IN (170,171)");

        // $chds = DB::select("SELECT
        //     COLL_REF_NO,INVOICE_NO,COL_ADJ_AMT,COL_ADJ_DATE
        //     from $collection_detail_sil WHERE COLL_REF_NO IN (459,460,1521,1376)");

        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (3589,4095)");
        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (191,647,14446)");
        // $chds = DB::select("SELECT * from $collection_detail_sil");
        if($chds){
            $chdAMT = 0;
            foreach ($chds as $detail) {
                echo 'REF-NO: ';
                echo $COLL_REF_NO = $detail->COLL_REF_NO;
                echo '<br>';
                echo '......';
                echo $invNO = $detail->INVOICE_NO;
                echo '----';
                $singleAMT = $detail->COL_ADJ_AMT;
                $singleAMT = str_replace(",","",$singleAMT);
                echo 'Coll-INV-AMT: ';
                echo $singleAMT = $CollInvAmt = (float)str_replace(",","",$singleAMT);

                echo '----';
                // echo $detail->COL_ADJ_DATE;
                echo $COLL_REF_DATE = DateTime::createFromFormat('m/d/Y', $detail->COL_ADJ_DATE)->format('d/m/Y');
                echo '----';
                echo '<br>';

                // $chdAMT += (float)$singleAMT;

                if($invNO){
                    $invInfos = DB::select("SELECT customer_id,id,total_amount_vat,due from $bill_processes WHERE INVOICE_NO='$invNO'");
                    if($invInfos){
                        foreach($invInfos as $invDetail){

                            $collMstTGCL = DB::selectOne("SELECT
                                COLL_REF_DATE,ENTERED_DATE,MODIFIED_DATE,MR_DATE,COLL_MODE,BANK_ID,RECEIVED_AMT,TDS_AMT,VAT_AMT,ADJ_AMT,
                                TOT_COLL_AMT,MR_NO,ADJ_REMARKS,DS_NO,DS_BANK_ID,BD_DATE,ENTERED_BY,CPDCR_NO,CPDCR_DATE
                                from $collection_master_sil where COLL_REF_NO='$COLL_REF_NO'");
                            $COLL_REF_DATE_MST = DateTime::createFromFormat('m/d/Y', $collMstTGCL->COLL_REF_DATE)->format('d/m/Y');
                            $COLL_REF_DATE_MST_CREATE = DateTime::createFromFormat('d/m/Y', $COLL_REF_DATE_MST)->format('Y-m-d h:i:s');

                            $checkCollMst = DB::selectOne("SELECT id,ref_no from $collections where ref_no='$COLL_REF_NO'");
                            if($checkCollMst){
                                $collMstID = $checkCollMst->id;

                            }else{
                                // insert to collection
                                if($collMstTGCL->ENTERED_DATE!=''){
                                    $ENTERED_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->ENTERED_DATE)->format('Y-m-d h:i:s');
                                }else{
                                    $ENTERED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                if($collMstTGCL->MODIFIED_DATE!=''){
                                    $MODIFIED_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->MODIFIED_DATE)->format('Y-m-d h:i:s');
                                }else{
                                    $MODIFIED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                $MR_DATE = null;
                                if($collMstTGCL->MR_DATE!=''){
                                    $MR_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->MR_DATE)->format('d/m/Y');
                                }
                                $INS_DATE = null;
                                if($collMstTGCL->CPDCR_DATE!=''){
                                    $INS_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->CPDCR_DATE)->format('d/m/Y');
                                }
                                $BD_DATE = null;
                                if($collMstTGCL->BD_DATE!=''){
                                    $BD_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->BD_DATE)->format('d/m/Y');
                                }

                                $RECEIVED_AMT = str_replace(",","",$collMstTGCL->RECEIVED_AMT);
                                $RECEIVED_AMT = (float)str_replace(",","",$RECEIVED_AMT);

                                $TDS_AMT = str_replace(",","",$collMstTGCL->TDS_AMT);
                                $TDS_AMT = (float)str_replace(",","",$TDS_AMT);

                                $VAT_AMT = str_replace(",","",$collMstTGCL->VAT_AMT);
                                $VAT_AMT = (float)str_replace(",","",$VAT_AMT);

                                $ADJ_AMT = str_replace(",","",$collMstTGCL->ADJ_AMT);
                                $ADJ_AMT = (float)str_replace(",","",$ADJ_AMT);

                                $TOT_COLL_AMT = str_replace(",","",$collMstTGCL->TOT_COLL_AMT);
                                $TOT_COLL_AMT = (float)str_replace(",","",$TOT_COLL_AMT);

                                $collectionData = array(
                                    'ref_no' => $COLL_REF_NO,
                                    'customer_id' => $invDetail->customer_id,
                                    'coll_date' => $COLL_REF_DATE_MST,
                                    'bank_deposit_date' => $BD_DATE,
                                    'instrument_mode' => $collMstTGCL->COLL_MODE,
                                    'instrument_no' => $collMstTGCL->CPDCR_NO,
                                    'instrument_date' => $INS_DATE,
                                    'bank_id' => $collMstTGCL->BANK_ID,
                                    'received_amount' => $RECEIVED_AMT,
                                    'tds_amount' => $TDS_AMT,
                                    'vat_amount' => $VAT_AMT,
                                    'adjust_amount' => $ADJ_AMT,
                                    'total_coll_amount' => $TOT_COLL_AMT,
                                    'advance_amount' => 0,
                                    'money_receipt_no' => $collMstTGCL->MR_NO,
                                    'money_receipt_date' => $MR_DATE,
                                    'adj_remarks' => $collMstTGCL->ADJ_REMARKS,
                                    'bank_slip_no' => $collMstTGCL->DS_NO,
                                    'dep_bank_id' => $collMstTGCL->DS_BANK_ID,
                                    'created_by' => $collMstTGCL->ENTERED_BY,
                                    'created_at' => $ENTERED_DATE,
                                    'updated_at' => $MODIFIED_DATE
                                );

                                DB::table($collections)->insert([$collectionData]);
                                $collMstID = DB::getPdo()->lastInsertId();
                            }

                            // echo '...............';
                            // echo 'Bill PK: ';
                            // echo $BillPK = $invDetail->id;
                            // echo '-----';
                            // echo 'Bill-Process-Amt: ';
                            // echo $BillProcessAmt = (float)$invDetail->total_amount_vat;
                            // echo '-----';

                            $BillPK = $invDetail->id;
                            $BillProcessAmt = (float)$invDetail->total_amount_vat;
                            $DueAmt = (float)$invDetail->due;

                            if($CollInvAmt >= $DueAmt){
                                $CollChdAmt = $DueAmt;
                                $newDue = 0;
                                $CollInvAmt = $CollInvAmt - $DueAmt;
                            }else{
                                $CollChdAmt = $CollInvAmt;
                                $newDue = $DueAmt - $CollInvAmt;
                                $CollInvAmt = 0;
                            }

                            $set_due = ['due' => $newDue];
                            DB::table($bill_processes)
                              ->where('id', $BillPK)
                              ->update($set_due);

                            if($CollChdAmt>0){
                                $chdData = array(
                                    'collection_id' => $collMstID,
                                    'ref_no' => $COLL_REF_NO,
                                    'bill_process_id' => $BillPK,
                                    'invoice_no' => $invNO,
                                    'collection_amount' => $CollChdAmt,
                                    'collection_date' => $COLL_REF_DATE_MST,
                                    'created_at' => $COLL_REF_DATE_MST_CREATE,
                                    'updated_at' => $COLL_REF_DATE_MST_CREATE
                                );

                                DB::table($collection_chds)->insert([$chdData]);
                            }

                            // $totalCollInfo = DB::selectOne("SELECT SUM(CAST(REPLACE(collection_amount, ',', '') AS DECIMAL(15, 2))) AS TotalColl
                            //     FROM $collection_chds WHERE bill_process_id = $BillPK");

                            // $DueAmt = $BillProcessAmt - $totalCollInfo->TotalColl;


                            // echo '----Remain-Coll-Inv-Amt: ';
                            // echo $CollInvAmt;
                            // echo '----Collection-Chd-Amt: ';
                            // echo $CollChdAmt;
                            // echo '----Due-Amt: ';
                            // echo $DueAmt;
                            // echo '<br>';

                        }
                    }
                }
            }
        }
    }

    public function collMigrationStore(){

        exit;

        // $collection_master_sil = 'aa_coll_mst_tgcl';
        // $collection_detail_sil = 'aa_coll_dtl_tgcl';
        // $bill_processes = 'aa_bill_processes';
        // $collections = 'aa_coll';
        // $collection_chds = 'aa_coll_chd';

        set_time_limit(0);

        // Define temp connection for this function only
        config([
            'database.connections.sil_billing_on_process' => [
                'driver' => 'mysql',
                'host' => '10.21.0.101',
                'database' => 'sil_billing_on_process',
                'username' => 'billing_user',  // change if needed
                'password' => 'Billing_usr@321',      // change if needed
                'charset' => 'utf8mb4',
                'collation' => 'utf8mb4_unicode_ci',
            ]
        ]);

        // Use the new DB connection
        $db = DB::connection('sil_billing_on_process');

        // $collection_master_sil = 'collection_master_sil_333';
        // $collection_detail_sil = 'collection_detail_sil_333';
        // $bill_processes = 'xxxx_bill_processes';
        // $collections = 'xxxx_coll';
        // $collection_chds = 'xxxx_coll_chd';

        // $collection_adjust_table = 'xxxx_coll_chd';

        // $collection_master_sil = 'collection_master_sil_333';
        // $collection_detail_sil = 'collection_detail_sil_333';
        // $bill_processes = 'yyy_bill_processes';
        // $collections = 'yyy_coll';
        // $collection_chds = 'yyy_coll_chd';

        // $collection_adjust_table = 'yyy_coll_adjust';


        $collection_master_sil = 'collection_master_receive_sil_444';
        $collection_detail_sil = 'collection_detail_sil_444';
        $bill_processes = 'zzzz_bill_processes';
        $collections = 'zzzz_coll';
        $collection_chds = 'zzzz_coll_chd';

        $collection_adjust_table = 'zzzz_coll_adjust';

        $chds = $db->select("SELECT
            COLL_REF_NO,INVOICE_NO,COL_ADJ_AMT,COL_ADJ_DATE
            from $collection_detail_sil");
            // from $collection_detail_sil WHERE COLL_REF_NO IN (3837,4083)");
            // from $collection_detail_sil WHERE COLL_REF_NO IN (170,171)");

        // $chds = DB::select("SELECT
        //     COLL_REF_NO,INVOICE_NO,COL_ADJ_AMT,COL_ADJ_DATE
        //     from $collection_detail_sil WHERE COLL_REF_NO IN (459,460,1521,1376)");

        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (3589,4095)");
        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (191,647,14446)");
        // $chds = DB::select("SELECT * from $collection_detail_sil");
        if($chds){
            $chdAMT = 0;
            foreach ($chds as $detail) {
                echo 'REF-NO: ';
                echo $COLL_REF_NO = $detail->COLL_REF_NO;
                echo '<br>';
                echo '......';
                echo $invNO = $detail->INVOICE_NO;
                echo '----';
                $singleAMT = $detail->COL_ADJ_AMT;
                $singleAMT = str_replace(",","",$singleAMT);
                echo 'Coll-INV-AMT: ';
                echo $singleAMT = $CollInvAmt = (float)str_replace(",","",$singleAMT);

                echo '----';
                // echo $detail->COL_ADJ_DATE;
                // echo $COLL_REF_DATE = Carbon::createFromFormat('m/d/Y', $detail->COL_ADJ_DATE)->format('d/m/Y');------------
                echo $COLL_REF_DATE = DateTime::createFromFormat('m/d/Y', $detail->COL_ADJ_DATE)->format('d/m/Y');
                echo '----';
                echo '<br>';

                // $chdAMT += (float)$singleAMT;

                if($invNO){
                    $invInfos = $db->select("SELECT customer_id,id,total_amount_vat,due from $bill_processes WHERE INVOICE_NO='$invNO'");
                    if($invInfos){
                        foreach($invInfos as $invDetail){

                            $collMstTGCL = $db->selectOne("SELECT
                                COLL_REF_DATE,ENTERED_DATE,MODIFIED_DATE,MR_DATE,COLL_MODE,BANK_ID,RECEIVED_AMT,TDS_AMT,VAT_AMT,ADJ_AMT,
                                TOT_COLL_AMT,MR_NO,ADJ_REMARKS,BD_DATE,DS_NO,DS_BANK_ID,ENTERED_BY,CPDCR_NO,CPDCR_DATE
                                from $collection_master_sil where COLL_REF_NO='$COLL_REF_NO'");
                            // $COLL_REF_DATE_MST = Carbon::createFromFormat('m/d/Y', $collMstTGCL->COLL_REF_DATE)->format('d/m/Y');------------
                            // $COLL_REF_DATE_MST_CREATE = Carbon::createFromFormat('d/m/Y', $COLL_REF_DATE_MST)->format('Y-m-d h:i:s');----------

                            $COLL_REF_DATE_MST = DateTime::createFromFormat('m/d/Y', $collMstTGCL->COLL_REF_DATE)->format('d/m/Y');
                            $COLL_REF_DATE_MST_CREATE = DateTime::createFromFormat('d/m/Y', $COLL_REF_DATE_MST)->format('Y-m-d') . ' 10:10:10';

                            $DEPOSIT_DATE_MST = null;
                            if (!empty($collMstTGCL->BD_DATE)) {
                                // $DEPOSIT_DATE_MST = Carbon::createFromFormat('m/d/Y', $collMstTGCL->BD_DATE)->format('d/m/Y');----------
                                $DEPOSIT_DATE_MST = DateTime::createFromFormat('m/d/Y', $collMstTGCL->BD_DATE)->format('d/m/Y');
                            }

                            $checkCollMst = $db->selectOne("SELECT id,ref_no from $collections where ref_no='$COLL_REF_NO'");
                            if($checkCollMst){
                                $collMstID = $checkCollMst->id;

                            }else{
                                // insert to collection
                                if($collMstTGCL->ENTERED_DATE!=''){
                                    // $ENTERED_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->ENTERED_DATE)->format('Y-m-d h:i:s');-----------
                                    $ENTERED_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->ENTERED_DATE)->format('Y-m-d') . ' 10:10:10';
                                }else{
                                    $ENTERED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                if($collMstTGCL->MODIFIED_DATE!=''){
                                    // $MODIFIED_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->MODIFIED_DATE)->format('Y-m-d h:i:s');-------------
                                    $MODIFIED_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->MODIFIED_DATE)->format('Y-m-d') . ' 10:10:10';

                                }else{
                                    $MODIFIED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                if($collMstTGCL->MR_DATE!=''){
                                    // $MR_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->MR_DATE)->format('d/m/Y');----------
                                    $MR_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->MR_DATE)->format('d/m/Y');
                                }else{
                                    $MR_DATE = null;
                                }
                                if($collMstTGCL->CPDCR_DATE!=''){
                                    // $INS_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->CPDCR_DATE)->format('d/m/Y');---------
                                    $INS_DATE = DateTime::createFromFormat('m/d/Y', $collMstTGCL->CPDCR_DATE)->format('d/m/Y');

                                }else{
                                    $INS_DATE = null;
                                }
                                $collectionData = array(
                                    'ref_no' => $COLL_REF_NO,
                                    'customer_id' => $invDetail->customer_id,
                                    'coll_date' => $COLL_REF_DATE_MST,
                                    'bank_deposit_date' => $DEPOSIT_DATE_MST,
                                    'instrument_mode' => $collMstTGCL->COLL_MODE,
                                    'instrument_no' => $collMstTGCL->CPDCR_NO,
                                    'instrument_date' => $INS_DATE,
                                    'bank_id' => $collMstTGCL->BANK_ID,
                                    'received_amount' => ((float)$collMstTGCL->RECEIVED_AMT>0)? $collMstTGCL->RECEIVED_AMT:0,
                                    'tds_amount' => ((float)$collMstTGCL->TDS_AMT>0)? $collMstTGCL->TDS_AMT:0,
                                    'vat_amount' => ((float)$collMstTGCL->VAT_AMT>0)? $collMstTGCL->VAT_AMT:0,
                                    'adjust_amount' => ((float)$collMstTGCL->ADJ_AMT>0)? $collMstTGCL->ADJ_AMT:0,
                                    'total_coll_amount' => ((float)$collMstTGCL->TOT_COLL_AMT>0)? $collMstTGCL->TOT_COLL_AMT:0,
                                    'advance_amount' => 0,
                                    'money_receipt_no' => $collMstTGCL->MR_NO,
                                    'money_receipt_date' => $MR_DATE,
                                    'adj_remarks' => $collMstTGCL->ADJ_REMARKS,
                                    'bank_slip_no' => $collMstTGCL->DS_NO,
                                    'dep_bank_id' => $collMstTGCL->DS_BANK_ID,
                                    'created_by' => $collMstTGCL->ENTERED_BY,
                                    'created_at' => $ENTERED_DATE,
                                    'updated_at' => $MODIFIED_DATE
                                );

                                $db->table($collections)->insert([$collectionData]);
                                $collMstID = $db->getPdo()->lastInsertId();
                            }

                            // echo '...............';
                            // echo 'Bill PK: ';
                            // echo $BillPK = $invDetail->id;
                            // echo '-----';
                            // echo 'Bill-Process-Amt: ';
                            // echo $BillProcessAmt = (float)$invDetail->total_amount_vat;
                            // echo '-----';

                            $BillPK = $invDetail->id;
                            $BillProcessAmt = (float)$invDetail->total_amount_vat;
                            $DueAmt = (float)$invDetail->due;

                            if($CollInvAmt >= $DueAmt){
                                $CollChdAmt = $DueAmt;
                                $newDue = 0;
                                $CollInvAmt = $CollInvAmt - $DueAmt;

                                // store to adjust
                                if($CollInvAmt > 0){
                                    $adjustData = array(
                                        'collection_id'      => $collMstID,
                                        'ref_no'             => $COLL_REF_NO,
                                        'bill_process_id'    => $BillPK,
                                        'invoice_no'         => $invNO,
                                        'adjust_amount'      => $CollChdAmt,
                                        'collection_date'    => $COLL_REF_DATE_MST,
                                        'created_at'         => $COLL_REF_DATE_MST_CREATE,
                                        'updated_at'         => $COLL_REF_DATE_MST_CREATE
                                    );
                                    $db->table($collection_adjust_table)->insert($adjustData);
                                }


                            }else{
                                $CollChdAmt = $CollInvAmt;
                                $newDue = $DueAmt - $CollInvAmt;
                                $CollInvAmt = 0;
                            }

                            $set_due = ['due' => $newDue];
                            $db->table($bill_processes)
                              ->where('id', $BillPK)
                              ->update($set_due);

                            if($CollChdAmt>0){
                                $chdData = array(
                                    'collection_id' => $collMstID,
                                    'ref_no' => $COLL_REF_NO,
                                    'bill_process_id' => $BillPK,
                                    'invoice_no' => $invNO,
                                    'collection_amount' => $CollChdAmt,
                                    'collection_date' => $COLL_REF_DATE_MST,
                                    'created_at' => $COLL_REF_DATE_MST_CREATE,
                                    'updated_at' => $COLL_REF_DATE_MST_CREATE
                                );

                                $db->table($collection_chds)->insert([$chdData]);
                            }

                            // $totalCollInfo = DB::selectOne("SELECT SUM(CAST(REPLACE(collection_amount, ',', '') AS DECIMAL(15, 2))) AS TotalColl
                            //     FROM $collection_chds WHERE bill_process_id = $BillPK");

                            // $DueAmt = $BillProcessAmt - $totalCollInfo->TotalColl;


                            // echo '----Remain-Coll-Inv-Amt: ';
                            // echo $CollInvAmt;
                            // echo '----Collection-Chd-Amt: ';
                            // echo $CollChdAmt;
                            // echo '----Due-Amt: ';
                            // echo $DueAmt;
                            // echo '<br>';

                        }
                    }
                }
            }
        }
    }

    public function collMigrationStoreEnvSameDB(){

        // exit;

        // $collection_master_sil = 'aa_coll_mst_tgcl';
        // $collection_detail_sil = 'aa_coll_dtl_tgcl';
        // $bill_processes = 'aa_bill_processes';
        // $collections = 'aa_coll';
        // $collection_chds = 'aa_coll_chd';

        set_time_limit(0);

        // $collection_master_sil = 'collection_master_sil_333';
        // $collection_detail_sil = 'collection_detail_sil_333';
        // $bill_processes = 'xxxx_bill_processes';
        // $collections = 'xxxx_coll';
        // $collection_chds = 'xxxx_coll_chd';

        // $collection_adjust_table = 'xxxx_coll_chd';

        // $collection_master_sil = 'collection_master_sil_333';
        // $collection_detail_sil = 'collection_detail_sil_333';
        // $bill_processes = 'yyy_bill_processes';
        // $collections = 'yyy_coll';
        // $collection_chds = 'yyy_coll_chd';

        // $collection_adjust_table = 'yyy_coll_adjust';


        $collection_master_sil = 'collection_master_receive_sil_444';
        $collection_detail_sil = 'collection_detail_sil_444';
        $bill_processes = 'zzzz_bill_processes';
        $collections = 'zzzz_coll';
        $collection_chds = 'zzzz_coll_chd';

        $collection_adjust_table = 'zzzz_coll_adjust';

        $chds = DB::select("SELECT
            COLL_REF_NO,INVOICE_NO,COL_ADJ_AMT,COL_ADJ_DATE
            from $collection_detail_sil");
            // from $collection_detail_sil WHERE COLL_REF_NO IN (3837,4083)");
            // from $collection_detail_sil WHERE COLL_REF_NO IN (170,171)");

        // $chds = DB::select("SELECT
        //     COLL_REF_NO,INVOICE_NO,COL_ADJ_AMT,COL_ADJ_DATE
        //     from $collection_detail_sil WHERE COLL_REF_NO IN (459,460,1521,1376)");

        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (3589,4095)");
        // $chds = DB::select("SELECT * from $collection_detail_sil WHERE COLL_REF_NO IN (191,647,14446)");
        // $chds = DB::select("SELECT * from $collection_detail_sil");
        if($chds){
            $chdAMT = 0;
            foreach ($chds as $detail) {
                echo 'REF-NO: ';
                echo $COLL_REF_NO = $detail->COLL_REF_NO;
                echo '<br>';
                echo '......';
                echo $invNO = $detail->INVOICE_NO;
                echo '----';
                $singleAMT = $detail->COL_ADJ_AMT;
                $singleAMT = str_replace(",","",$singleAMT);
                echo 'Coll-INV-AMT: ';
                echo $singleAMT = $CollInvAmt = (float)str_replace(",","",$singleAMT);

                echo '----';
                // echo $detail->COL_ADJ_DATE;
                echo $COLL_REF_DATE = Carbon::createFromFormat('m/d/Y', $detail->COL_ADJ_DATE)->format('d/m/Y');
                echo '----';
                echo '<br>';

                // $chdAMT += (float)$singleAMT;

                if($invNO){
                    $invInfos = DB::select("SELECT customer_id,id,total_amount_vat,due from $bill_processes WHERE INVOICE_NO='$invNO'");
                    if($invInfos){
                        foreach($invInfos as $invDetail){

                            $collMstTGCL = DB::selectOne("SELECT
                                COLL_REF_DATE,ENTERED_DATE,MODIFIED_DATE,MR_DATE,COLL_MODE,BANK_ID,RECEIVED_AMT,TDS_AMT,VAT_AMT,ADJ_AMT,
                                TOT_COLL_AMT,MR_NO,ADJ_REMARKS,BD_DATE,DS_NO,DS_BANK_ID,ENTERED_BY,CPDCR_NO,CPDCR_DATE
                                from $collection_master_sil where COLL_REF_NO='$COLL_REF_NO'");
                            $COLL_REF_DATE_MST = Carbon::createFromFormat('m/d/Y', $collMstTGCL->COLL_REF_DATE)->format('d/m/Y');
                            $COLL_REF_DATE_MST_CREATE = Carbon::createFromFormat('d/m/Y', $COLL_REF_DATE_MST)->format('Y-m-d h:i:s');

                            $DEPOSIT_DATE_MST = null;
                            if (!empty($collMstTGCL->BD_DATE)) {
                                $DEPOSIT_DATE_MST = Carbon::createFromFormat('m/d/Y', $collMstTGCL->BD_DATE)->format('d/m/Y');
                            }

                            $checkCollMst = DB::selectOne("SELECT id,ref_no from $collections where ref_no='$COLL_REF_NO'");
                            if($checkCollMst){
                                $collMstID = $checkCollMst->id;

                            }else{
                                // insert to collection
                                if($collMstTGCL->ENTERED_DATE!=''){
                                    $ENTERED_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->ENTERED_DATE)->format('Y-m-d h:i:s');
                                }else{
                                    $ENTERED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                if($collMstTGCL->MODIFIED_DATE!=''){
                                    $MODIFIED_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->MODIFIED_DATE)->format('Y-m-d h:i:s');
                                }else{
                                    $MODIFIED_DATE = $COLL_REF_DATE_MST_CREATE;
                                }
                                if($collMstTGCL->MR_DATE!=''){
                                    $MR_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->MR_DATE)->format('d/m/Y');
                                }else{
                                    $MR_DATE = null;
                                }
                                if($collMstTGCL->CPDCR_DATE!=''){
                                    $INS_DATE = Carbon::createFromFormat('m/d/Y', $collMstTGCL->CPDCR_DATE)->format('d/m/Y');
                                }else{
                                    $INS_DATE = null;
                                }
                                $collectionData = array(
                                    'ref_no' => $COLL_REF_NO,
                                    'customer_id' => $invDetail->customer_id,
                                    'coll_date' => $COLL_REF_DATE_MST,
                                    'bank_deposit_date' => $DEPOSIT_DATE_MST,
                                    'instrument_mode' => $collMstTGCL->COLL_MODE,
                                    'instrument_no' => $collMstTGCL->CPDCR_NO,
                                    'instrument_date' => $INS_DATE,
                                    'bank_id' => $collMstTGCL->BANK_ID,
                                    'received_amount' => ((float)$collMstTGCL->RECEIVED_AMT>0)? $collMstTGCL->RECEIVED_AMT:0,
                                    'tds_amount' => ((float)$collMstTGCL->TDS_AMT>0)? $collMstTGCL->TDS_AMT:0,
                                    'vat_amount' => ((float)$collMstTGCL->VAT_AMT>0)? $collMstTGCL->VAT_AMT:0,
                                    'adjust_amount' => ((float)$collMstTGCL->ADJ_AMT>0)? $collMstTGCL->ADJ_AMT:0,
                                    'total_coll_amount' => ((float)$collMstTGCL->TOT_COLL_AMT>0)? $collMstTGCL->TOT_COLL_AMT:0,
                                    'advance_amount' => 0,
                                    'money_receipt_no' => $collMstTGCL->MR_NO,
                                    'money_receipt_date' => $MR_DATE,
                                    'adj_remarks' => $collMstTGCL->ADJ_REMARKS,
                                    'bank_slip_no' => $collMstTGCL->DS_NO,
                                    'dep_bank_id' => $collMstTGCL->DS_BANK_ID,
                                    'created_by' => $collMstTGCL->ENTERED_BY,
                                    'created_at' => $ENTERED_DATE,
                                    'updated_at' => $MODIFIED_DATE
                                );

                                DB::table($collections)->insert([$collectionData]);
                                $collMstID = DB::getPdo()->lastInsertId();
                            }

                            // echo '...............';
                            // echo 'Bill PK: ';
                            // echo $BillPK = $invDetail->id;
                            // echo '-----';
                            // echo 'Bill-Process-Amt: ';
                            // echo $BillProcessAmt = (float)$invDetail->total_amount_vat;
                            // echo '-----';

                            $BillPK = $invDetail->id;
                            $BillProcessAmt = (float)$invDetail->total_amount_vat;
                            $DueAmt = (float)$invDetail->due;

                            if($CollInvAmt >= $DueAmt){
                                $CollChdAmt = $DueAmt;
                                $newDue = 0;
                                $CollInvAmt = $CollInvAmt - $DueAmt;

                                // store to adjust
                                if($CollInvAmt > 0){
                                    $adjustData = array(
                                        'collection_id'      => $collMstID,
                                        'ref_no'             => $COLL_REF_NO,
                                        'bill_process_id'    => $BillPK,
                                        'invoice_no'         => $invNO,
                                        'adjust_amount'      => $CollChdAmt,
                                        'collection_date'    => $COLL_REF_DATE_MST,
                                        'created_at'         => $COLL_REF_DATE_MST_CREATE,
                                        'updated_at'         => $COLL_REF_DATE_MST_CREATE
                                    );
                                    DB::table($collection_adjust_table)->insert($adjustData);
                                }


                            }else{
                                $CollChdAmt = $CollInvAmt;
                                $newDue = $DueAmt - $CollInvAmt;
                                $CollInvAmt = 0;
                            }

                            $set_due = ['due' => $newDue];
                            DB::table($bill_processes)
                              ->where('id', $BillPK)
                              ->update($set_due);

                            if($CollChdAmt>0){
                                $chdData = array(
                                    'collection_id' => $collMstID,
                                    'ref_no' => $COLL_REF_NO,
                                    'bill_process_id' => $BillPK,
                                    'invoice_no' => $invNO,
                                    'collection_amount' => $CollChdAmt,
                                    'collection_date' => $COLL_REF_DATE_MST,
                                    'created_at' => $COLL_REF_DATE_MST_CREATE,
                                    'updated_at' => $COLL_REF_DATE_MST_CREATE
                                );

                                DB::table($collection_chds)->insert([$chdData]);
                            }

                            // $totalCollInfo = DB::selectOne("SELECT SUM(CAST(REPLACE(collection_amount, ',', '') AS DECIMAL(15, 2))) AS TotalColl
                            //     FROM $collection_chds WHERE bill_process_id = $BillPK");

                            // $DueAmt = $BillProcessAmt - $totalCollInfo->TotalColl;


                            // echo '----Remain-Coll-Inv-Amt: ';
                            // echo $CollInvAmt;
                            // echo '----Collection-Chd-Amt: ';
                            // echo $CollChdAmt;
                            // echo '----Due-Amt: ';
                            // echo $DueAmt;
                            // echo '<br>';

                        }
                    }
                }
            }
        }
    }

    public function collMigrationStoreDevineOptimizedNotChecked() {
        // Remove exit statement to allow execution
        exit;

        set_time_limit(0);

        $collection_master_sil = 'Collection_Master_SIL';
        $collection_detail_sil = 'Collection_Detail_SIL';
        $bill_processes = 'zzzz';
        $collections = 'zzzz_coll';
        $collection_chds = 'zzzz_coll_chd';

        // 1. Fetch all collection details at once with a single query
        $chds = DB::select("SELECT COLL_REF_NO, INVOICE_NO, COL_ADJ_AMT, COL_ADJ_DATE FROM $collection_detail_sil");

        if (empty($chds)) {
            return; // Early return if no records found
        }

        // 2. Extract all unique COLL_REF_NOs to fetch collection masters in bulk
        $collRefNos = [];
        $invoiceNos = [];
        foreach ($chds as $detail) {
            $collRefNos[] = $detail->COLL_REF_NO;
            if (!empty($detail->INVOICE_NO)) {
                $invoiceNos[] = $detail->INVOICE_NO;
            }
        }
        $uniqueCollRefNos = array_unique($collRefNos);
        $uniqueInvoiceNos = array_unique($invoiceNos);

        // 3. Fetch all collection masters in one query
        $collMasters = [];
        if (!empty($uniqueCollRefNos)) {
            $placeholders = implode(',', array_fill(0, count($uniqueCollRefNos), '?'));
            $collMastersResults = DB::select("SELECT
                COLL_REF_NO, COLL_REF_DATE, ENTERED_DATE, MODIFIED_DATE, MR_DATE,
                COLL_MODE, BANK_ID, RECEIVED_AMT, TDS_AMT, VAT_AMT, ADJ_AMT,
                TOT_COLL_AMT, MR_NO, ADJ_REMARKS, DS_NO, DS_BANK_ID, BD_DATE,
                ENTERED_BY, CPDCR_NO, CPDCR_DATE
                FROM $collection_master_sil
                WHERE COLL_REF_NO IN ($placeholders)", $uniqueCollRefNos);

            foreach ($collMastersResults as $master) {
                $collMasters[$master->COLL_REF_NO] = $master;
            }
        }

        // 4. Fetch all invoice information in one query
        $invoiceInfos = [];
        if (!empty($uniqueInvoiceNos)) {
            $placeholders = implode(',', array_fill(0, count($uniqueInvoiceNos), '?'));
            $invoiceResults = DB::select("SELECT
                INVOICE_NO, customer_id, id, total_amount_vat, due
                FROM $bill_processes
                WHERE INVOICE_NO IN ($placeholders)", $uniqueInvoiceNos);

            foreach ($invoiceResults as $invoice) {
                $invoiceInfos[$invoice->INVOICE_NO][] = $invoice;
            }
        }

        // 5. Fetch existing collection masters to avoid duplicates
        $existingCollections = [];
        if (!empty($uniqueCollRefNos)) {
            $placeholders = implode(',', array_fill(0, count($uniqueCollRefNos), '?'));
            $existingResults = DB::select("SELECT id, ref_no FROM $collections WHERE ref_no IN ($placeholders)", $uniqueCollRefNos);

            foreach ($existingResults as $existing) {
                $existingCollections[$existing->ref_no] = $existing->id;
            }
        }

        // 6. Prepare batch updates and inserts
        $billUpdates = [];
        $collectionInserts = [];
        $collectionChildInserts = [];
        $processedCollections = [];

        // Helper function to clean amount strings and convert to float
        $cleanAmount = function($amount) {
            return (float)str_replace(',', '', $amount);
        };

        // Helper function to format dates
        $formatDate = function($dateStr, $fromFormat, $toFormat) {
            if (empty($dateStr)) return null;
            return DateTime::createFromFormat($fromFormat, $dateStr)->format($toFormat);
        };

        // Process each collection detail
        foreach ($chds as $detail) {
            $COLL_REF_NO = $detail->COLL_REF_NO;
            $invNO = $detail->INVOICE_NO;
            $CollInvAmt = $cleanAmount($detail->COL_ADJ_AMT);
            $COLL_REF_DATE = $formatDate($detail->COL_ADJ_DATE, 'm/d/Y', 'd/m/Y');

            // Skip if no invoice number or no invoice info found
            if (empty($invNO) || !isset($invoiceInfos[$invNO])) {
                continue;
            }

            // Skip if no collection master found
            if (!isset($collMasters[$COLL_REF_NO])) {
                continue;
            }

            $collMstTGCL = $collMasters[$COLL_REF_NO];
            $COLL_REF_DATE_MST = $formatDate($collMstTGCL->COLL_REF_DATE, 'm/d/Y', 'd/m/Y');
            $COLL_REF_DATE_MST_CREATE = $formatDate($COLL_REF_DATE_MST, 'd/m/Y', 'Y-m-d h:i:s');

            // Get or create collection master ID
            if (isset($existingCollections[$COLL_REF_NO])) {
                $collMstID = $existingCollections[$COLL_REF_NO];
            } else {
                // Only process each collection once
                if (!isset($processedCollections[$COLL_REF_NO])) {
                    $ENTERED_DATE = !empty($collMstTGCL->ENTERED_DATE)
                        ? $formatDate($collMstTGCL->ENTERED_DATE, 'm/d/Y', 'Y-m-d h:i:s')
                        : $COLL_REF_DATE_MST_CREATE;

                    $MODIFIED_DATE = !empty($collMstTGCL->MODIFIED_DATE)
                        ? $formatDate($collMstTGCL->MODIFIED_DATE, 'm/d/Y', 'Y-m-d h:i:s')
                        : $COLL_REF_DATE_MST_CREATE;

                    $MR_DATE = !empty($collMstTGCL->MR_DATE)
                        ? $formatDate($collMstTGCL->MR_DATE, 'm/d/Y', 'd/m/Y')
                        : null;

                    $INS_DATE = !empty($collMstTGCL->CPDCR_DATE)
                        ? $formatDate($collMstTGCL->CPDCR_DATE, 'm/d/Y', 'd/m/Y')
                        : null;

                    $BD_DATE = !empty($collMstTGCL->BD_DATE)
                        ? $formatDate($collMstTGCL->BD_DATE, 'm/d/Y', 'd/m/Y')
                        : null;

                    // Process all invoice details for this collection to get customer_id
                    $customer_id = null;
                    foreach ($invoiceInfos[$invNO] as $invDetail) {
                        $customer_id = $invDetail->customer_id;
                        break;
                    }

                    if ($customer_id) {
                        $collectionData = [
                            'ref_no' => $COLL_REF_NO,
                            'customer_id' => $customer_id,
                            'coll_date' => $COLL_REF_DATE_MST,
                            'bank_deposit_date' => $BD_DATE,
                            'instrument_mode' => $collMstTGCL->COLL_MODE,
                            'instrument_no' => $collMstTGCL->CPDCR_NO,
                            'instrument_date' => $INS_DATE,
                            'bank_id' => $collMstTGCL->BANK_ID,
                            'received_amount' => $cleanAmount($collMstTGCL->RECEIVED_AMT),
                            'tds_amount' => $cleanAmount($collMstTGCL->TDS_AMT),
                            'vat_amount' => $cleanAmount($collMstTGCL->VAT_AMT),
                            'adjust_amount' => $cleanAmount($collMstTGCL->ADJ_AMT),
                            'total_coll_amount' => $cleanAmount($collMstTGCL->TOT_COLL_AMT),
                            'advance_amount' => 0,
                            'money_receipt_no' => $collMstTGCL->MR_NO,
                            'money_receipt_date' => $MR_DATE,
                            'adj_remarks' => $collMstTGCL->ADJ_REMARKS,
                            'bank_slip_no' => $collMstTGCL->DS_NO,
                            'dep_bank_id' => $collMstTGCL->DS_BANK_ID,
                            'created_by' => $collMstTGCL->ENTERED_BY,
                            'created_at' => $ENTERED_DATE,
                            'updated_at' => $MODIFIED_DATE
                        ];

                        $collectionInserts[] = $collectionData;
                        $processedCollections[$COLL_REF_NO] = true;
                    }
                }

                // We'll set temporary IDs and update them after batch insert
                $collMstID = "temp_" . $COLL_REF_NO;
            }

            // Process each invoice for this collection detail
            foreach ($invoiceInfos[$invNO] as $invDetail) {
                $BillPK = $invDetail->id;
                $BillProcessAmt = (float)$invDetail->total_amount_vat;
                $DueAmt = (float)$invDetail->due;
                $remainingCollInvAmt = $CollInvAmt;

                if ($remainingCollInvAmt >= $DueAmt) {
                    $CollChdAmt = $DueAmt;
                    $newDue = 0;
                    $remainingCollInvAmt -= $DueAmt;
                } else {
                    $CollChdAmt = $remainingCollInvAmt;
                    $newDue = $DueAmt - $remainingCollInvAmt;
                    $remainingCollInvAmt = 0;
                }

                // Add to bill updates batch
                $billUpdates[] = [
                    'id' => $BillPK,
                    'due' => $newDue
                ];

                // Add to collection child inserts batch if amount > 0
                if ($CollChdAmt > 0) {
                    $collectionChildInserts[] = [
                        'collection_id' => $collMstID,
                        'ref_no' => $COLL_REF_NO,
                        'bill_process_id' => $BillPK,
                        'invoice_no' => $invNO,
                        'collection_amount' => $CollChdAmt,
                        'collection_date' => $COLL_REF_DATE_MST,
                        'created_at' => $COLL_REF_DATE_MST_CREATE,
                        'updated_at' => $COLL_REF_DATE_MST_CREATE
                    ];
                }

                // If we've used up all the collection amount, break out of the loop
                if ($remainingCollInvAmt <= 0) {
                    break;
                }
            }
        }

        // 7. Execute batch operations within a transaction for data integrity
        DB::beginTransaction();
        try {
            // Insert collections in batch
            if (!empty($collectionInserts)) {
                $collectionIds = [];
                foreach (array_chunk($collectionInserts, 100) as $chunk) {
                    DB::table($collections)->insert($chunk);
                    $lastId = DB::getPdo()->lastInsertId();

                    // Calculate IDs for all inserted records
                    for ($i = 0; $i < count($chunk); $i++) {
                        $refNo = $chunk[$i]['ref_no'];
                        $collectionIds["temp_" . $refNo] = $lastId + $i;
                        $existingCollections[$refNo] = $lastId + $i;
                    }
                }

                // Update temporary collection IDs in child inserts
                foreach ($collectionChildInserts as &$child) {
                    if (is_string($child['collection_id']) && isset($collectionIds[$child['collection_id']])) {
                        $child['collection_id'] = $collectionIds[$child['collection_id']];
                    }
                }
            }

            // Update bills in batch
            if (!empty($billUpdates)) {
                foreach (array_chunk($billUpdates, 100) as $chunk) {
                    foreach ($chunk as $update) {
                        DB::table($bill_processes)
                            ->where('id', $update['id'])
                            ->update(['due' => $update['due']]);
                    }
                }
            }

            // Insert collection children in batch
            if (!empty($collectionChildInserts)) {
                foreach (array_chunk($collectionChildInserts, 100) as $chunk) {
                    DB::table($collection_chds)->insert($chunk);
                }
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            // Log error or handle exception
            throw $e;
        }
    }

    public function attentionStoreToChild(){

        exit;
        $customers = DB::select("SELECT *
            FROM billing_customers
            WHERE attention_person IS NOT NULL AND attention_person != ''");

        foreach($customers as $customer){
            $timestamp = time();
            $formattedDate = date('Y-m-d H:i:s', $timestamp);

            $data = array(
                'customer_id' => $customer->id,
                'attention_name' => $customer->attention_person,
                'selected' => 1,
                'status' => 'Active',
                'created_by' => auth()->id(),
                'updated_by' => auth()->id(),
                'created_at' => $formattedDate,
                'updated_at' => $formattedDate
            );

            DB::table('billing_attentions')->insert([$data]);
            $attChdId = DB::getPdo()->lastInsertId();

            $update_person = ['attention_person' => $attChdId];
            DB::table("billing_customers")
                ->where('id', $customer->id)
                ->update($update_person);
        }

    }

    public function attentionStoreToBill(){

        exit;
        $billTbl = "billing_bills";
        $billProcessTbl = "bill_processes";

        $bills = DB::select("SELECT id,customer_id FROM $billTbl");
        if($bills){
            foreach ($bills as $bill) {
                $customerId = $bill->customer_id;
                $billID = $bill->id;

                $custInfo = DB::selectOne("SELECT attention_person FROM billing_customers WHERE id=$customerId");
                if($custInfo){
                    $attenID = $custInfo->attention_person;
                    $attenInfo = DB::selectOne("SELECT id,attention_name FROM billing_attentions WHERE id='$attenID'");

                    $attentionID = null;
                    $attentionName = null;

                    if($attenInfo){
                        $attentionID = $attenInfo->id;
                        $attentionName = $attenInfo->attention_name;
                    }

                    // Update in bills
                    $update_person = [
                        'attention_id' => $attentionID,
                        'attention_person' => $attentionName,
                    ];
                    DB::table($billTbl)
                        ->where('id', $billID)
                        ->update($update_person);
                }

            }
        }

    }

    public function deleteVoucherRange(Request $request){

        if ($request->isMethod('post')) {

            if($request->code==1200060){

                $token = session('api_token');
                $api_ip = env("API_IP");
                $company_number = env("COMPANY_NUMBER");

                // Define the range of voucher IDs
                $startId = $request->startId;
                $endId = $request->endId;

                // Loop through each voucher ID
                for ($voucherMstId = $startId; $voucherMstId <= $endId; $voucherMstId++) {
                    try {
                        // Make the API DELETE request
                        $response = Http::withHeaders([
                            'Authorization' => 'Bearer ' . $token,
                        ])->delete($api_ip.'/api/v1/deleteVoucher/'.$voucherMstId);

                        // Check for successful response
                        if ($response->successful()) {
                            echo "Deleted- $voucherMstId";
                        } else {
                            echo "Failed- $voucherMstId";
                        }
                        echo '<br>';
                    } catch (\Exception $e) {
                        // Handle any exceptions that may occur during the request
                        //echo "Error occurred while deleting Voucher ID $voucherMstId: " . $e->getMessage() . "\n";
                    }
                }

            }else{
                return redirect()->route('deleteVoucherRange')
                    ->with('error', 'Security code not found.');
            }
        }


        return view('collection.deleteVoucherRange');

    }

}
