<?php
namespace App\Http\Controllers;
use DB;
use Carbon\Carbon;
use App\Models\Bill;
use App\Models\GlHead;
use App\Models\Customer;
use App\Models\Settings;
use App\Models\Collection;
use App\Models\BillProcess;
use App\Models\CustomerBank;
use Illuminate\Http\Request;
use App\Models\CollectionChd;
use App\Models\DepositoryBank;
use App\Models\AdvancedCollection;
use App\Models\AdvancedCustomer;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;

class CollectionController extends Controller
{
    public function index()
    {
        if (!auth()->user()->can('View Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $collections = Collection::all();
        return view('collection.index', compact('collections'));
    }

    public function collectionList(Request $request)
    {
        if (!auth()->user()->can('View Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $perPage = $request->input('per_page', 25);

        $query = Collection::query();

        if ($request->filled('start_date')) {
            $startDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->start_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '>=', $startDate);
        }

        if ($request->filled('end_date')) {
            $endDate = Carbon::createFromFormat('d/m/Y', $this->normalizeDate($request->end_date))->format('Y-m-d');
            $query->where(DB::raw("STR_TO_DATE(coll_date, '%d/%m/%Y')"), '<=', $endDate);
        }

        if ($request->filled('instrument_mode')) {
            if($request->instrument_mode!='All'){
                $query->where('instrument_mode', $request->instrument_mode);
            }
        }

        // $collections = $query->get();

        $collections = $query->orderBy('id', 'desc')->paginate($perPage)->appends($request->except('page'));

        // $collections = Collection::all();
        return view('collection.index', compact('collections', 'perPage'));
    }

    public function collEntry()
    {
        if (!auth()->user()->can('Create Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $banks = CustomerBank::orderBy('bank_name')->get();
        $depBanks = [];
        $depCash = [];

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        // Bank GL Code
        $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
        if($getBankHead){
            $glCodeBank = $getBankHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeBank,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depBanks = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }

        // Cash GL Code
        $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
        if($getCashHead){
            $glCodeCash = $getCashHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeCash,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depCash = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }

        //get department list from api - salaquzzaman @ 05-06-2024
        $departments = [];
        try{
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        }catch (\Exception $e) {
            // Log::error($e->getMessage());
        }

        $distinctCustomerIds = BillProcess::where('payment_status','!=','Paid')->select('customer_id')->distinct()->pluck('customer_id');
        $customers = Customer::whereIn('id', $distinctCustomerIds)
            ->select('autoAcNo','customer_name')
            ->orderBy('customer_name', 'asc')
            ->get();

        return view('collection.entry', compact('customers','banks','depBanks','depCash','departments'));
    }

    public function collSearch($customerId=null){

        $bills = [];
        if($customerId && ($fndCustomer = Customer::where('autoAcNo',$customerId)->first())){
            $bills = DB::select("SELECT * FROM bill_processes
                    WHERE customer_id = $fndCustomer->id
                    AND (payment_status != 'Paid' || payment_status = '' || payment_status IS NULL)");
        }
        echo view('collection.result', compact('bills'));
    }

    public function userAdvanceAmount($customerId){
        $customerInfo = Customer::where('autoAcNo',$customerId)->first();
        $advAmount = AdvancedCollection::where('customer',$customerInfo->id)->sum('amount');
        echo $advAmount - $customerInfo->advance_adjust;
    }

    public function store(Request $request)
    {

        if (!auth()->user()->can('Create Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $validator = Validator::make($request->all(), [
            'ref_no' => 'required',
            'total_coll_amount' => 'required',
        ]);
        if ($validator->fails()) {
            return redirect()->route('coll.entry')
                         ->with('error', 'Missing required fields.');
        }

        $totalColAmt = $request->total_coll_amount;
        if(ceil($totalColAmt)<1){
            return redirect()->route('coll.entry')
                         ->with('error', 'Missing collection amount.');
        }

        $fndCustomer = Customer::where('autoAcNo',$request->customer_id)->first();
        $custPKID = $fndCustomer->id;

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $depDivCode = '';
        $depDivName = '';
        $cashDivCode = '';
        $cashDivName = '';

        if($request->instrument_mode=='Cheque'){

            // Bank GL Code
            // $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
            // if($getBankHead){
            //     $glCodeBank = $getBankHead->gl_code;

            //     //get bank list from api - salaquzzaman @ 01-06-2024
            //     try{
            //         $response = Http::withHeaders([
            //             'Authorization' => 'Bearer ' . $token,
            //         ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
            //             'groupCode' => $glCodeBank,
            //             'companyNo' => $company_number,
            //         ]);

            //         if ($response->successful()) {
            //             $responseData = $response->json();
            //             if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
            //                 $depBanks = json_encode($responseData['accHeadDetailsInfos']);

            //                 // Decode JSON data to PHP array
            //                 $accounts = json_decode($depBanks, true);
            //                 $result = null;

            //                 // Search for the specific account by autoAcNo
            //                 foreach ($accounts as $account) {
            //                     if ($account['autoAcNo'] == $request->dep_bank_id) {
            //                         $result = $account;
            //                         break;
            //                     }
            //                 }
            //                 $depDivCode = $result['divCode'];
            //                 $depDivName = $result['divName'];
            //             }
            //         }
            //     }catch (\Exception $e) {
            //         // Log::error($e->getMessage());
            //     }
            // }

            $depDivCode = $request->department_id;
            $depDivName = $request->depDivName;

        }else if($request->instrument_mode=='Cash'){

            // CASH GL Code
            // $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
            // if($getCashHead){
            //     $glCodeCash = $getCashHead->gl_code;

            //     //get cash list from api - salaquzzaman @ 22-08-2024
            //     try{
            //         $response = Http::withHeaders([
            //             'Authorization' => 'Bearer ' . $token,
            //         ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
            //             'groupCode' => $glCodeCash,
            //             'companyNo' => $company_number,
            //         ]);

            //         if ($response->successful()) {
            //             $responseData = $response->json();
            //             if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
            //                 $depCash = json_encode($responseData['accHeadDetailsInfos']);

            //                 // Decode JSON data to PHP array
            //                 $accounts = json_decode($depCash, true);
            //                 $result = null;

            //                 // Search for the specific account by autoAcNo
            //                 foreach ($accounts as $account) {
            //                     if ($account['autoAcNo'] == $request->dep_cash_id) {
            //                         $result = $account;
            //                         break;
            //                     }
            //                 }
            //                 $cashDivCode = $result['divCode'];
            //                 $cashDivName = $result['divName'];
            //             }
            //         }
            //     }catch (\Exception $e) {
            //         // Log::error($e->getMessage());
            //     }
            // }


            $cashDivCode = $request->department_id;
            $cashDivName = $request->depDivName;

        }else{

        }

        $cusDivCode = '';
        $cusDivName = '';

        // Customer GL Code
        // $getCusHead = GlHead::where('gl_subject',strtoupper('Accounts Receivable'))->first();
        // if($getCusHead){
        //     $glCodeCus = $getCusHead->gl_code;

        //     //get customer list from api - salaquzzaman @ 03-06-2024
        //     try{
        //         $response2 = Http::withHeaders([
        //             'Authorization' => 'Bearer ' . $token,
        //         ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
        //             'groupCode' => $glCodeCus,
        //             'companyNo' => $company_number,
        //         ]);

        //         if ($response2->successful()) {
        //             $responseData2 = $response2->json();
        //             if ($responseData2 && isset($responseData2['accHeadDetailsInfos'])) {
        //                 $customers = json_encode($responseData2['accHeadDetailsInfos']);

        //                 // Decode JSON data to PHP array
        //                 $accounts = json_decode($customers, true);
        //                 $result = null;

        //                 // Search for the specific account by autoAcNo
        //                 foreach ($accounts as $account) {
        //                     if ($account['autoAcNo'] == $fndCustomer->autoAcNo) {
        //                         $result = $account;
        //                         break;
        //                     }
        //                 }
        //                 $cusDivCode = $result['divCode'];
        //                 $cusDivName = $result['divName'];
        //             }
        //         }
        //     }catch (\Exception $e) {
        //         // Log::error($e->getMessage());
        //     }
        // }


        $cusDivCode = $request->department_id;
        $cusDivName = $request->depDivName;

        $taxAutoAcNo = '';
        $taxDivCode = '';
        $taxDivName = '';

        if(round($request->tds_amount)>0){

            // TDS/TAX Amount
            $getTaxHead = GlHead::where('gl_subject',strtoupper('TAX'))->first();
            if($getTaxHead){
                $glCodeTax = $getTaxHead->gl_code;

                //get tds/tax code details from api - salaquzzaman @ 08-06-2024
                try{
                    $response3 = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                        'groupCode' => $glCodeTax,
                        'companyNo' => $company_number,
                    ]);

                    if ($response3->successful()) {
                        $responseData3 = $response3->json();
                        if ($responseData3 && isset($responseData3['accHeadDetailsInfo'])) {

                            $taxAutoAcNo = $responseData3['accHeadDetailsInfo']['autoAcNo'];
                            // $taxDivCode = $responseData3['accHeadDetailsInfo']['divCode'];
                            // $taxDivName = $responseData3['accHeadDetailsInfo']['divName'];
                            $taxDivCode = $request->department_id;
                            $taxDivName = $request->depDivName;
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }


        $vatAutoAcNo = '';
        $vatDivCode = '';
        $vatDivName = '';

        if(round($request->vat_amount)>0){

            // TDS/TAX Amount
            $getVatHead = GlHead::where('gl_subject',strtoupper('VAT'))->first();
            if($getVatHead){
                $glCodeVat = $getVatHead->gl_code;

                //get vat code details from api - salaquzzaman @ 08-06-2024
                try{
                    $response4 = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                        'groupCode' => $glCodeVat,
                        'companyNo' => $company_number,
                    ]);

                    if ($response4->successful()) {
                        $responseData4 = $response4->json();
                        if ($responseData4 && isset($responseData4['accHeadDetailsInfo'])) {

                            $vatAutoAcNo = $responseData4['accHeadDetailsInfo']['autoAcNo'];
                            // $vatDivCode = $responseData4['accHeadDetailsInfo']['divCode'];
                            // $vatDivName = $responseData4['accHeadDetailsInfo']['divName'];
                            $vatDivCode = $request->department_id;
                            $vatDivName = $request->depDivName;
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }


        $adjAutoAcNo = '';
        $adjDivCode = '';
        $adjDivName = '';

        if(round($request->adjust_amount)>0){

            // TDS/TAX Amount
            $getAdjHead = GlHead::where('gl_subject',strtoupper('Provission For Bad Debts'))->first();
            if($getAdjHead){
                $glCodeAdj = $getAdjHead->gl_code;

                //get adjust details from api - salaquzzaman @ 08-06-2024
                try{
                    $response5 = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/autoAcNoByGroupCode', [
                        'groupCode' => $glCodeAdj,
                        'companyNo' => $company_number,
                    ]);

                    if ($response5->successful()) {
                        $responseData5 = $response5->json();
                        if ($responseData5 && isset($responseData5['accHeadDetailsInfo'])) {

                            $adjAutoAcNo = $responseData5['accHeadDetailsInfo']['autoAcNo'];
                            // $adjDivCode = $responseData5['accHeadDetailsInfo']['divCode'];
                            // $adjDivName = $responseData5['accHeadDetailsInfo']['divName'];
                            $adjDivCode = $request->department_id;
                            $adjDivName = $request->depDivName;
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }

        ($taxAutoAcNo)? $TTX_NA = '||Collection from '.$fndCustomer->customer_name : $TTX_NA = '';
        ($taxAutoAcNo)? $TTX_A = ','.$taxAutoAcNo : $TTX_A = '';
        ($taxAutoAcNo)? $TTX_D = ',0' : $TTX_D = '';
        ($taxAutoAcNo)? $TTX_C = ','.$request->tds_amount : $TTX_C = '';
        ($taxAutoAcNo)? $TTX_N = '||'.$taxDivName : $TTX_N = '';
        ($taxAutoAcNo)? $TTX_CO = ','.$taxDivCode : $TTX_CO = '';

        ($vatAutoAcNo)? $VAAT_NA = '||Collection from '.$fndCustomer->customer_name : $VAAT_NA = '';
        ($vatAutoAcNo)? $VAAT_A = ','.$vatAutoAcNo : $VAAT_A = '';
        ($vatAutoAcNo)? $VAAT_D = ',0' : $VAAT_D = '';
        ($vatAutoAcNo)? $VAAT_C = ','.$request->vat_amount : $VAAT_C = '';
        ($vatAutoAcNo)? $VAAT_N = '||'.$vatDivName : $VAAT_N = '';
        ($vatAutoAcNo)? $VAAT_CO = ','.$vatDivCode : $VAAT_CO = '';

        ($adjAutoAcNo)? $ADJ_NA = '||Collection from '.$fndCustomer->customer_name : $ADJ_NA = '';
        ($adjAutoAcNo)? $ADJ_A = ','.$adjAutoAcNo : $ADJ_A = '';
        ($adjAutoAcNo)? $ADJ_D = ',0' : $ADJ_D = '';
        ($adjAutoAcNo)? $ADJ_C = ','.$request->adjust_amount : $ADJ_C = '';
        ($adjAutoAcNo)? $ADJ_N = '||'.$adjDivName : $ADJ_N = '';
        ($adjAutoAcNo)? $ADJ_CO = ','.$adjDivCode : $ADJ_CO = '';

        $rcvAmt = (float)($request->received_amount);
        $tdsAmt = (float)($request->tds_amount);
        $vatAmt = (float)($request->vat_amount);
        $adjAmt = (float)($request->adjust_amount);

        $advance_amount = (float)$request->advance_amount;
        $deducted_amount = (float)$request->deducted_amount;
        $total_coll_amount = (float)$request->total_coll_amount;

        $voucherMode = 'JV';
        if($total_coll_amount>$deducted_amount){
            if($request->instrument_mode=='Cheque'){
                $voucherMode = 'BRV';
            }else if($request->instrument_mode=='Cash'){
                $voucherMode = 'CRV';
            }else{}
        }

        // Advance adjustment
        $ADV_NA = '';
        $ADV_A = '';
        $ADV_D = '';
        $ADV_C = '';
        $ADV_N = '';
        $ADV_CO = '';

        if($request->set_adjustment=='1' && $deducted_amount>0 && $advance_amount>0){
            $advCustomer = AdvancedCustomer::where('customerhead_id',$fndCustomer->id)->first();

            $ADV_NA = "||Collection from $fndCustomer->customer_name";
            $ADV_A = ','.$advCustomer->customeradvancedhead_autoAcNo;
            $ADV_D = ','.$deducted_amount;
            $ADV_C = ',0';
            $ADV_N = '||'.$request->depDivName;
            $ADV_CO = ','.$request->department_id;
        }

        $remaining_amount = 0;
        if($total_coll_amount>$deducted_amount){
            $remaining_amount = $total_coll_amount - $deducted_amount;
        }

        if($request->instrument_mode=='Cheque'){
            $BANK_NA = "||Collection from $fndCustomer->customer_name";
            $BANK_A = ','.$request->dep_bank_id;
            $BANK_D = ','.$remaining_amount;
            $BANK_C = ',0';
            $BANK_N = '||'.$depDivName;
            $BANK_CO = ','.$depDivCode;
        }else if($request->instrument_mode=='Cash'){
            $BANK_NA = "||Collection from $fndCustomer->customer_name";
            $BANK_A = ','.$request->dep_cash_id;
            $BANK_D = ','.$remaining_amount;
            $BANK_C = ',0';
            $BANK_N = '||'.$cashDivName;
            $BANK_CO = ','.$cashDivCode;
        }else{
            $BANK_NA = '';
            $BANK_A = '';
            $BANK_D = '';
            $BANK_C = '';
            $BANK_N = '';
            $BANK_CO = '';
        }

        $customerAMT = 0;
        if($rcvAmt<1 && $tdsAmt<1 && $vatAmt<1 && $adjAmt>0){
            // only adjust amount/ bad depts
            $BANK_NA = '';
            $BANK_A = '';
            $BANK_D = '';
            $BANK_C = '';
            $BANK_N = '';
            $BANK_CO = '';
            $customerAMT = $adjAmt;
        }


        $narration = "Collection from $fndCustomer->customer_name$BANK_NA$TTX_NA$VAAT_NA$ADJ_NA$ADV_NA";
        $autoAcNo = $fndCustomer->autoAcNo.$BANK_A.$TTX_A.$VAAT_A.$ADJ_A.$ADV_A;
        $debitAmt = "$customerAMT$BANK_D$TTX_D$VAAT_D$ADJ_D$ADV_D";
        $creditAmt = $rcvAmt.$BANK_C.$TTX_C.$VAAT_C.$ADJ_C.$ADV_C;
        $divName = "$cusDivName$BANK_N$TTX_N$VAAT_N$ADJ_N$ADV_N";
        $divCode = "$cusDivCode$BANK_CO$TTX_CO$VAAT_CO$ADJ_CO$ADV_CO";

        $formColDate = '';
        $formInsDate = '';
        if($request->coll_date){
            $formColDate = Settings::dateMaker($request->coll_date,'d/m/Y','Y-m-d');
        }
        if($request->bank_deposit_date){
            $formInsDate = Settings::dateMaker($request->bank_deposit_date,'d/m/Y','Y-m-d');
        }

        echo '<pre>';
        var_dump([
            'voucherDt' => $formColDate,
            'voucherFlag' => $voucherMode,
            'narration' =>  $narration,
            'verifyFlag' => 'N',
            'autoAcNo' => $autoAcNo,
            'debitAmt' => $debitAmt,
            'creditAmt' => $creditAmt,
            'companyNo' => $company_number,
            'divName' => $divName,
            'divCode' => $divCode,
            'chqNo' => ($request->instrument_mode=='Cheque')? $request->bank_slip_no:'',
            'chqDt' => ($request->instrument_mode=='Cheque')? $formInsDate:'',
        ]);
        exit;

        try{
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->post($api_ip.'/api/v1/voucherEntry/saveVoucherInfo', [
                'voucherDt' => $formColDate,
                'voucherFlag' => $voucherMode,
                'narration' =>  $narration,
                'verifyFlag' => 'N',
                'autoAcNo' => $autoAcNo,
                'debitAmt' => $debitAmt,
                'creditAmt' => $creditAmt,
                'companyNo' => $company_number,
                'divName' => $divName,
                'divCode' => $divCode,
                'chqNo' => ($request->instrument_mode=='Cheque')? $request->bank_slip_no:'',
                'chqDt' => ($request->instrument_mode=='Cheque')? $formInsDate:'',
            ]);
            if($response->successful()){
                // store to db
                $collection                   = new Collection;
                $collection->ref_no           = $request->ref_no;
                $collection->customer_id      = $custPKID;
                $collection->coll_date        = $request->coll_date;
                $collection->instrument_mode  = $request->instrument_mode;
                $collection->instrument_no    = $request->instrument_no;
                $collection->instrument_date  = $request->instrument_date;
                $collection->bank_id          = $request->bank_id;
                $collection->received_amount  = $request->received_amount;
                $collection->tds_amount       = $request->tds_amount;
                $collection->vat_amount       = $request->vat_amount;
                $collection->adjust_amount    = $request->adjust_amount;
                $collection->total_coll_amount   = $request->total_coll_amount;
                $collection->excess_amount       = $request->total_remaining;
                $collection->money_receipt_no    = $request->money_receipt_no;
                $collection->money_receipt_date  = $request->money_receipt_date;
                $collection->bank_slip_no        = $request->bank_slip_no;
                $collection->bank_deposit_date   = $request->bank_deposit_date;
                $collection->dep_bank_id         = ($request->instrument_mode=='Cheque')? $request->dep_bank_id:$request->dep_cash_id;
                $collection->adj_remarks         = $request->adj_remarks;
                $collection->created_by          = Auth::user()->id;

                if($collection->save()){
                    $billId = $request->billId;
                    if($billId){
                        $k = 0;
                        foreach($billId as $blID){

                            // Bill Process Update
                            $bill = BillProcess::find($blID);
                            $bill->payment_status     = $request->flagStatus[$k];
                            $bill->save();

                            $billInfo = Bill::find($bill->bill_id);


                            // Collection Amount Insert
                            if(floatval($request->collectionAmt[$k])>0){
                                $colChd                    = new CollectionChd;
                                $colChd->collection_id     = $collection->id;
                                $colChd->bill_process_id   = $blID;
                                $colChd->project_id        = $billInfo->project_id;
                                $colChd->collection_amount = $request->collectionAmt[$k];
                                $colChd->save();
                            }
                            $k = $k + 1;
                        }
                    }

                    $fndCustomer->advance_adjust = $fndCustomer->advance_adjust + $deducted_amount;
                    $fndCustomer->update();


                    return redirect()->route('collection.index')
                            ->with('success', 'Collection received successfully.');
                }

            }else{
                return redirect()->route('coll.entry')
                        ->with('error', 'Collection not received.');
            }

        }catch (\Exception $e) {
            if($e->getMessage()){
                return redirect()->route('collection.index')
                        ->with('error', 'Collection not received.');
            }
        }

    }

    public function show($collID)
    {
        if (!auth()->user()->can('View Collection')) {
            abort(403, 'Sorry !! You are not authorized!');
        }

        $depBanks = [];
        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $pay = Collection::find($collID);
        if($pay->instrument_mode=='Cheque'){

            // Bank GL Code
            $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
            if($getBankHead){
                $glCodeBank = $getBankHead->gl_code;

                //get deposit bank list from api - salaquzzaman @ 05-06-2024
                try{
                    $response = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                        'groupCode' => $glCodeBank,
                        'companyNo' => $company_number,
                    ]);

                    if ($response->successful()) {
                        $responseData = $response->json();
                        if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                            $depBanks = json_encode($responseData['accHeadDetailsInfos']);
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }else{ //cash

            // CASH GL Code
            $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
            if($getCashHead){
                $glCodeCash = $getCashHead->gl_code;

                //get cash list from api - salaquzzaman @ 22-08-2024
                try{
                    $response = Http::withHeaders([
                        'Authorization' => 'Bearer ' . $token,
                    ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                        'groupCode' => $glCodeCash,
                        'companyNo' => $company_number,
                    ]);

                    if ($response->successful()) {
                        $responseData = $response->json();
                        if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                            $depBanks = json_encode($responseData['accHeadDetailsInfos']);
                        }
                    }
                }catch (\Exception $e) {
                    // Log::error($e->getMessage());
                }
            }

        }

        // $payChd = CollectionChd::where('collection_id',$collID)->get();
        return view('collection.show', compact('pay','depBanks'));
    }

    private function normalizeDate($date)
    {
        $formats = [
            'd/m/y',
            'd/m/Y',
            'd-M-Y',
            'd-M-y'
        ];

        foreach ($formats as $format) {
            try {
                $parsedDate = \Carbon\Carbon::createFromFormat($format, $date);
                return $parsedDate->format('d/m/Y');
            } catch (\Exception $e) {
            }
        }

        return $date;
    }




    public function advancedCollection()
    {
        $advancedcollections = AdvancedCollection::all();

        $customerIds = $advancedcollections->pluck('customer')->unique()->toArray();
        $customerNames = DB::table('billing_customers')->whereIn('id', $customerIds)->pluck('customer_name', 'id')->toArray();

        $bankIds = $advancedcollections->pluck('bank')->unique()->toArray();
        $bankNames = DB::table('depository_banks')->whereIn('id', $bankIds)->pluck('bank_name', 'id')->toArray();

        return view('collection.advancedcollectionindex', compact('advancedcollections', 'customerNames', 'bankNames'));
    }


    public function advancedcollectionCreate()
    {

        $customers = [];

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        // Customer GL Code
        $getCusHead = GlHead::where('gl_subject',strtoupper('Accounts Receivable'))->first();
        if($getCusHead){
            $glCodeCus = $getCusHead->gl_code;

            //get customer list from api - salaquzzaman @ 03-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeCus,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $customers = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }


        $banks = CustomerBank::orderBy('bank_name')->get();
        $depBanks = [];
        $depCash = [];

        // Bank GL Code
        $getBankHead = GlHead::where('gl_subject',strtoupper('Bank'))->first();
        if($getBankHead){
            $glCodeBank = $getBankHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeBank,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depBanks = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }

        // Cash GL Code
        $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
        if($getCashHead){
            $glCodeCash = $getCashHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeCash,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depCash = $responseData['accHeadDetailsInfos'];
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }


        //get department list from api - salaquzzaman @ 05-06-2024
        $departments = [];
        try{
            $response2 = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip.'/api/v1/departments/all');

            if ($response2->successful()) {
                $responseData2 = $response2->json();
                if ($responseData2) {
                    $departments = $responseData2;
                }
            }
        }catch (\Exception $e) {
            // Log::error($e->getMessage());
        }

        return view('collection.advancedcollectioncreate', compact('customers', 'banks', 'depBanks','departments','depCash'));
    }



    public function advancedcollectionStore(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'amount' => 'required|numeric|min:1',

        ]);

        if ($validator->fails()) {
            return redirect()->route('advanced.collection.create')
                ->with('error', 'Missing required fields.');
        }

        // Check if customer exists
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        } else {
            // Create new customer
            $customer_name = $request->customerName;
            $email = Settings::genRandString(6) . '@example.com';
            $autoAcNo = $request->customer_id;
            $phoneNo = '0' . mt_rand(11111, 999999);
            $cusInfo = [
                'customer_name' => $customer_name,
                'phone' => $phoneNo,
                'billing_name' => $customer_name,
                'billing_phone' => $phoneNo,
                'billing_email' => $email,
                'account_name' => $customer_name . ' A/C',
                'account_phone' => $phoneNo,
                'account_email' => $email,
                'status' => 'Active',
                'autoAcNo' => $autoAcNo,
            ];

            Log::info('Creating new customer:', $cusInfo); // Log the customer info

            $newCust = Customer::create($cusInfo + ['created_by' => auth()->id()]);
            $custPKID = $newCust->id;
        }

        $advData = AdvancedCustomer::where('customerhead_id',$custPKID)->first();
        if(!$advData){
            return redirect()->route('advanced.collection.create')
                ->with('error', 'Please setup customer against advance customer setup.');
        }

        $bankPKID = null;
        if($request->instrument_mode=='Bank'){

            $fndBank = DepositoryBank::where('bank_id', $request->dep_bank_id)->first();
            if ($fndBank) {
                $bankPKID = $fndBank->id;
                $bankAutoAcNo = $fndBank->bank_id;
            } else {
                // Create new bank
                $bank_name = $request->bankName;
                $autoAcNo = $request->dep_bank_id;
                $phoneNo = '0' . mt_rand(11111, 99999);
                $holder = 'User-' . mt_rand(11111, 999999);
                $bankInfo = [
                    'holder_name' => $holder,
                    'bank_name' => $bank_name,
                    'account_number' => 0,
                    'opening_balance' => 0,
                    'phone' => $phoneNo,
                    'bank_id' => $autoAcNo,
                    'divCode' => $request->bankDivCode,
                    'created_by' => \auth()->id(),
                    'is_active' => 1
                ];


                Log::info('Creating new bank:', $bankInfo); // Log the bank info

                $bankNew = DepositoryBank::create($bankInfo);
                $bankPKID = $bankNew->id;
                $bankAutoAcNo = $bankNew->bank_id;
            }
        }

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $setAutoAC = ($request->instrument_mode=='Bank')? $bankAutoAcNo:$request->dep_cash_id;
        $customerAgnstAC = $advData->customeradvancedhead_autoAcNo;

        $narration = 'Advance Collection||Advance Collection';
        $autoAcNo = $setAutoAC.','.$customerAgnstAC;
        $debitAmt = $request->amount.',0';
        $creditAmt = '0,'.$request->amount;
        $divName = "$request->depDivName||$request->depDivName";
        $divCode = "$request->department_id,$request->department_id";

        try{
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->post($api_ip.'/api/v1/voucherEntry/saveVoucherInfo', [
                'voucherDt' => date('Y-m-d'),
                'voucherFlag' => 'JV',
                'narration' =>  $narration,
                'verifyFlag' => 'N',
                'autoAcNo' => $autoAcNo,
                'debitAmt' => $debitAmt,
                'creditAmt' => $creditAmt,
                'companyNo' => $company_number,
                'divName' => $divName,
                'divCode' => $divCode
            ]);
            if($response->successful()){
                // store to db
                AdvancedCollection::create([
                    'customer' => $custPKID,
                    'amount' => $request->amount,
                    'instrument_mode' => $request->instrument_mode,
                    'department_id' => $request->department_id,
                    'depDivName' => $request->depDivName,
                    'bank' => ($request->instrument_mode=='Bank')? $bankPKID:$request->dep_cash_id
                ]);

                return redirect()->route('advanced.collection')->with('success', 'Advanced collection created successfully.');


            }else{
                return redirect()->route('advanced.collection')
                        ->with('error', 'Collection not created.');
            }

        }catch (\Exception $e) {
            if($e->getMessage()){
                return redirect()->route('advanced.collection')
                        ->with('error', 'Collection not created.');
            }
        }

    }

    public function advancedcollectionView($id)
    {
        $advancedcollection = AdvancedCollection::findOrFail($id);

        $customerIds = collect([$advancedcollection->customer]);
        $customerNames = DB::table('billing_customers')->whereIn('id', $customerIds)->pluck('customer_name', 'id')->toArray();

        $bankIds = collect([$advancedcollection->bank]);
        $bankNames = DB::table('depository_banks')->whereIn('id', $bankIds)->pluck('bank_name', 'id')->toArray();

        $token = session('api_token');
        $api_ip = env("API_IP");
        $company_number = env("COMPANY_NUMBER");

        $depCash = [];

        // Cash GL Code
        $getCashHead = GlHead::where('gl_subject',strtoupper('Cash'))->first();
        if($getCashHead){
            $glCodeCash = $getCashHead->gl_code;

            //get bank list from api - salaquzzaman @ 01-06-2024
            try{
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $token,
                ])->get($api_ip.'/api/v1/integrate/searchByGroupCode', [
                    'groupCode' => $glCodeCash,
                    'companyNo' => $company_number,
                ]);

                if ($response->successful()) {
                    $responseData = $response->json();
                    if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                        $depCash = json_encode($responseData['accHeadDetailsInfos']);
                    }
                }
            }catch (\Exception $e) {
                // Log::error($e->getMessage());
            }
        }

        return view('collection.advancedcollectionview', compact('advancedcollection', 'customerNames', 'bankNames','depCash'));
    }



public function advancedcollectionEdit($id)
{
    $collection = AdvancedCollection::findOrFail($id);

    $customers = [];
    $token = session('api_token');
    $api_ip = env("API_IP");
    $company_number = env("COMPANY_NUMBER");

    $getCusHead = GlHead::where('gl_subject', strtoupper('Accounts Receivable'))->first();
    if ($getCusHead) {
        $glCodeCus = $getCusHead->gl_code;

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip . '/api/v1/integrate/searchByGroupCode', [
                'groupCode' => $glCodeCus,
                'companyNo' => $company_number,
            ]);

            if ($response->successful()) {
                $responseData = $response->json();
                if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                    $customers = $responseData['accHeadDetailsInfos'];
                }
            }
        } catch (\Exception $e) {
        }
    }

    $autoAcNo = '';
    if($collection->customer && ($custInfo = Customer::find($collection->customer))){

        $autoAcNo = $custInfo->autoAcNo;
    }



    $depBanks = [];
    $getBankHead = GlHead::where('gl_subject', strtoupper('Bank'))->first();
    if ($getBankHead) {
        $glCodeBank = $getBankHead->gl_code;

        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
            ])->get($api_ip . '/api/v1/integrate/searchByGroupCode', [
                'groupCode' => $glCodeBank,
                'companyNo' => $company_number,
            ]);

            if ($response->successful()) {
                $responseData = $response->json();
                if ($responseData && isset($responseData['accHeadDetailsInfos'])) {
                    $depBanks = $responseData['accHeadDetailsInfos'];
                }
            }
        } catch (\Exception $e) {

        }
    }


    $bankautoAcNo = '';

    if($collection->bank && ($banksInfo = DepositoryBank::find($collection->bank))){
        $bankautoAcNo = $banksInfo->bank_id;
    }

    return view('collection.advancedcollectionedit', compact('customers', 'depBanks', 'collection', 'autoAcNo', 'bankautoAcNo'));
}



    public function advancedcollectionUpdate(Request $request, $id)
    {
        // Validate the request
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required',
            'amount' => 'required|numeric|min:1',
            'dep_bank_id' => 'required',
        ]);

        if ($validator->fails()) {
            return redirect()->route('advanced.collection.edit', $id)
                ->with('error', 'Missing required fields.')
                ->withErrors($validator)
                ->withInput();
        }

        // Check if customer exists
        $fndCustomer = Customer::where('autoAcNo', $request->customer_id)->first();
        if ($fndCustomer) {
            $custPKID = $fndCustomer->id;
            $autoAcNo = $fndCustomer->autoAcNo;
        } else {
            // Create new customer
            $customer_name = $request->customerName;
            $email = Settings::genRandString(6) . '@example.com';
            $autoAcNo = $request->customer_id;
            $phoneNo = '0' . mt_rand(11111, 999999);
            $cusInfo = [
                'customer_name' => $customer_name,
                'phone' => $phoneNo,
                'billing_name' => $customer_name,
                'billing_phone' => $phoneNo,
                'billing_email' => $email,
                'account_name' => $customer_name . ' A/C',
                'account_phone' => $phoneNo,
                'account_email' => $email,
                'status' => 'Active',
                'autoAcNo' => $autoAcNo,
            ];

            Log::info('Creating new customer:', $cusInfo); // Log the customer info

            $newCust = Customer::create($cusInfo + ['created_by' => auth()->id()]);
            $custPKID = $newCust->id;
        }

        // Check if bank exists
        $fndBank = DepositoryBank::where('bank_id', $request->dep_bank_id)->first();
        if ($fndBank) {
            $bankPKID = $fndBank->id;
            $bankAutoAcNo = $fndBank->bank_id;
        } else {
            // Create new bank
            $bank_name = $request->bankName;
            $autoAcNo = $request->dep_bank_id;
            $phoneNo = '0' . mt_rand(11111, 99999);
            $holder = 'User-' . mt_rand(11111, 999999);
            $bankInfo = [
                'holder_name' => $holder,
                'bank_name' => $bank_name,
                'account_number' => 0,
                'opening_balance' => 0,
                'phone' => $phoneNo,
                'bank_id' => $autoAcNo,
                'divCode' => $request->bankDivCode,
                'created_by' => \auth()->id(),
                'is_active' => 1
            ];

            Log::info('Creating new bank:', $bankInfo); // Log the bank info

            $bankNew = DepositoryBank::create($bankInfo);
            $bankPKID = $bankNew->id;
            $bankAutoAcNo = $bankNew->bank_id;
        }

        // Update the advanced collection
        $advancedCollection = AdvancedCollection::findOrFail($id);
        $advancedCollection->update([
            'customer' => $custPKID,
            'amount' => $request->amount,
            'bank' => $bankPKID,
        ]);

        return redirect()->route('advanced.collection')->with('success', 'Advanced collection updated successfully.');
    }


    public function advancedcollectionDelete($id)
    {
        $advancedcollections = AdvancedCollection::findOrFail($id);
        $advancedcollections->delete();

        return redirect()->route('advanced.collection', $advancedcollections) ->with('success', 'Advanced Collection deleted successfully');
    }

}
